package com.xebialabs.deployit.documentation;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.xebialabs.commons.html.Element;
import com.xebialabs.commons.html.HtmlWriter;
import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import java.io.PrintWriter;
import java.util.*;

import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;

/**
 * Html generator for CI documentation.
 */
public class CiReferenceHtmlWriter extends HtmlWriter {

    private Set<Type> typesInDocument = new HashSet<>();

    public CiReferenceHtmlWriter(PrintWriter writer) {
        super(writer);
    }

    public void startCiReference() {
        div().cssClass("ci-reference").writeOpen();
    }

    public void endCiReference() {
        div().writeClose();
    }

    public void tocHeader() {
        h2("CI Reference").cssClass("ci-toc-category").write();
        h3("Configuration Item Overview").cssClass("ci-details-title").write();
    }

    public void startToc() {
        table().cssClass("ci-toc").writeOpen();
        rowHeader("CI", "Description").cssClass("odd ci-toc-header").write();
    }

    public void endToc() {
        table().writeClose();
    }

    public void tocEntry(Descriptor ci, String firstSentence, int row) {

        Element type = link("#" + ci.getType(), ci.getType());
        if (ci.isVirtual()) {
            type = type.cssClass("virtual-type");
        }

        row(
                type,
                firstSentence
        ).cssClass(row % 2 == 0 ? "even" : "odd").write();
    }

    public void detailsHeader() {
        h3("Configuration Item Details").cssClass("ci-details-title").write();
    }

    public void detailsSeparator() {
        hr().write();
    }

    public void ciDescription(String description) {
        p(description).cssClass("ci-description").write();
    }

    public void ciDetailTitle(Type type) {
        anchor(type.toString()).write();
        h4(type).cssClass("ci-detail-title").write();
    }

    public void ciTypeInfo(Descriptor ci) {
        if (ci.getSuperClasses().isEmpty()) {
            return;
        }

        Element table = table().cssClass("ci-relations-table");

        if (ci.isVirtual()) {
            table.add(tr(th("Virtual Type").attribute("colspan", "2")));
        }

        if (ci.getSuperClasses().size() > 1) {
            table.add(generateTypeRow(ci.getSuperClasses(), "Type&nbsp;Hierarchy", " >> "));
        }

        Set<Type> interfaces = new LinkedHashSet<Type>(ci.getInterfaces());
        interfaces.remove(Type.valueOf(ConfigurationItem.class));
        if (!ci.getInterfaces().isEmpty()) {
            table.add(generateTypeRow(ci.getInterfaces(), "Interfaces", ", "));
        }

        table.write();
    }

    private Element generateTypeRow(Collection<Type> types, String header, String separator) {
        Element row = rowHeader(header);
        Element cell = td();
        row.add(cell);

        int i = 0;
        for (Type type : types) {
            cell.add(render(type));
            i++;
            if (i < types.size()) {
                cell.add(separator);
            }
        }

        return row;
    }

    private String render(Type type) {
        if (typesInDocument.contains(type)) {
            return link("#" + type.toString(), type).toString();
        } else {
            return type.toString();
        }
    }

    public void ciProperties(Collection<PropertyDescriptor> properties, String category) {
        if (properties.isEmpty()) {
            return;
        }

        Element table = table().cssClass("ci-table");
        table.add(rowHeader("", "", category).cssClass("odd ci-prop-header"));

        int row = 0;
        for (PropertyDescriptor property : properties) {
            String css = row % 2 == 0 ? "even" : "odd";
            table.add(getPropertyInfoRow(property).cssClass(css));
            table.add(getPropertyDescriptionRow(property).cssClass(css));
            row++;
        }

        table.write();
    }

    private Element getPropertyInfoRow(PropertyDescriptor descriptor) {

        // Start row
        Element row = tr();

        // Inspection
        row.add(td(getInspectionIcon(descriptor)).cssClass("icon-cell"));

        // Required or not
        row.add(td(getRequiredIcon(descriptor)).cssClass("icon-cell"));

        // Property name
        Element cell = div().cssClass("ci-property-info");
        row.add(td(cell).attribute("style", "width: 100%"));
        cell.add(span(descriptor.getName()).cssClass("ci-property-name"));
        cell.add(span(": "));

        // Kind
        Element kind = span(descriptor.getKind()).cssClass("ci-property-kind");
        cell.add(kind);
        if (descriptor.getKind() == PropertyKind.ENUM) {
            kind.add(" ", Arrays.toString(descriptor.getEnumValues().toArray()));
        }
        if (descriptor.getKind() == PropertyKind.CI || descriptor.getKind() == PropertyKind.SET_OF_CI || descriptor.getKind() == LIST_OF_CI) {
            kind.add("&lt;", render(descriptor.getReferencedType()), "&gt;");
        }

        // Default value
        if (descriptor.getDefaultValue() != null) {
            cell.add("&nbsp;=&nbsp;");
            cell.add(span(descriptor.getDefaultValue()).cssClass("ci-property-default"));
        }

        return row;
    }

    private Element getRequiredIcon(PropertyDescriptor descriptor) {
        Element icon = div("&nbsp;").cssClass("icon");
        if (descriptor.isRequired()) {
            icon.cssClass("icon required").attribute("title", "Required property");
        }

        return icon;
    }

    private Element getInspectionIcon(PropertyDescriptor descriptor) {
        Element icon = div("&nbsp;").cssClass("icon");

        if (descriptor.isInspectionProperty()) {
            if (descriptor.isRequiredForInspection()) {
                icon.cssClass("icon inspection-required").attribute("title", "Inspection property (required)");
            } else {
                icon.cssClass("icon inspection").attribute("title", "Inspection property");
            }
        }

        return icon;
    }

    private Element getPropertyDescriptionRow(PropertyDescriptor descriptor) {
        return row("", "", div(descriptor.getDescription()).cssClass("ci-property-desc"));
    }

    public void startControlTask(String category) {
        table().cssClass("ci-properties").writeOpen();
        rowHeader("Control task", "Parameter CI", "Attributes", "Description").cssClass("odd ci-prop-header").write();
    }

    public void endControlTask() {
        table().writeClose();
    }

    public void controlTaskMethod(MethodDescriptor method, int row) {
        Type type = method.getParameterObjectType();
        row(
                span(method.getName()).cssClass("ci-property-name"),
                type == null ? "" : link("#" + type.toString(), type.toString()),
                p(listAtributes(method)),
                p(method.getDescription()).cssClass("ci-property-desc")
        ).cssClass(row % 2 == 0 ? "even" : "odd").write();
    }

    private String listAtributes(final MethodDescriptor method) {
        Joiner joiner = Joiner.on(", ").skipNulls();
        return joiner.join(Iterables.transform(method.getAttributes().entrySet(), new Function<Map.Entry<String, String>, Object>() {
            @Override
            public Object apply(final java.util.Map.Entry<String, String> input) {
                return new StringBuilder().append(input.getKey()).append(" = ").append(input.getValue()).toString();
            }
        }));
    }

    public void categoryHeader(String category) {
        h4(category).cssClass("ci-toc-category").write();
    }

    public void setCis(List<Descriptor> cis) {
        typesInDocument = Sets.newLinkedHashSet();
        for (Descriptor ci : cis) {
            typesInDocument.add(ci.getType());
        }
    }

    public void startDocument() {
        html().writeOpen();
        head(
                title("CI Reference"),
                linkCss("deployit.css"),
                linkCss("ci-reference-api.css")
        ).write();
        body().writeOpen();
    }

    public void endDocument() {
        body().writeClose();
        html().writeClose();
    }
}
