/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.spi.orchestration;

import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecificationWithDependencies;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * An Orchestrator converts a DeltaSpecification into a Orchestration, which can then be resolved to result in steps.
 * Using the {@link com.xebialabs.deployit.engine.spi.orchestration.Orchestrations} utilities, it is easy to assimilate the Orchestration.
 */
public interface Orchestrator {

    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.TYPE)
    public @interface Metadata {
        public String name();

        public String description() default "";
    }

    @Deprecated // TODO this one is used from the inteface itself?!
    default Orchestration orchestrate(DeltaSpecification specification) {
        throw new UnsupportedOperationException("This method is deprecated and should not be used");
    }

    default Orchestration orchestrate(DeltaSpecificationWithDependencies specificationWithDependencies) {
        List<DeltaSpecification> allDeltaSpecifications = specificationWithDependencies.getAllDeltaSpecifications();
        if (allDeltaSpecifications.size() == 1) {
            return orchestrate(allDeltaSpecifications.get(0));
        } else {
            Logger logger = LoggerFactory.getLogger(getClass());
            logger.warn("This orchestrator is not implemented for application dependencies. Please implement method orchestrate(DeltaSpecificationWithDependencies specificationWithDependencies)");
            logger.warn("Falling back to sequential by application dependency orchestration");
            Stream<Orchestration> orchestrationStream = allDeltaSpecifications.stream().map(this::orchestrate);
            String description = allDeltaSpecifications.stream()
                    .map(ds -> ds.getDeployedApplication().getName())
                    .collect(Collectors.joining(", ", "Deploying applications ", ""));
            return Orchestrations.serial(description, orchestrationStream.collect(Collectors.toList()));
        }
    }
}
