/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.spi.artifact.resolution;

import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * A resolver for {@link com.xebialabs.deployit.plugin.api.udm.artifact.Artifact} files. In order for it to be found by
 * XL Deploy it should be annotated with the enclosed {@link com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver.Resolver}
 * annotation.
 */
public interface ArtifactResolver {

    /**
     * Denote which protocols the {@link com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver} supports.
     */
    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.TYPE)
    public @interface Resolver {
        String[] protocols();
    }

    /**
     * This method is only called with a {@link com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact} that has a
     * {@link com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact#getFileUri()} URI that contains (one of) the URI schemes that the
     * {@link com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver.Resolver} annotation specifies.
     *
     * @param artifact  The {@link com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact} for which we need to
     *                  resolve a {@link com.xebialabs.deployit.engine.spi.artifact.resolution.ResolvedArtifactFile}
     *                  against its {@link com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact#getFileUri()}.
     * @return The {@link com.xebialabs.deployit.engine.spi.artifact.resolution.ResolvedArtifactFile} from which you can obtain the {@link java.io.InputStream}.
     * @throws CannotLocateArtifactException when the file pointed to by the artifact could not be found at the indicated URI.
     */
    ResolvedArtifactFile resolveLocation(SourceArtifact artifact);

    /**
     * Validate whether the {@link com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact#getFileUri()} {@link java.net.URI} is correct with regards to the scheme specific part of the defined scheme.
     * Only the resolver that "knows about" this scheme is able to do this.
     *
     * <em>NOTE:</em> There is no guarantee that when this method returns <pre>true</pre>, that the location actually resolves
     * to a file, or the file is present. It is implementation specific whether this is checked with this call.
     *
     * @param artifact The {@link com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact} which needs to be
     *                 validated with regards to its {@link com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact#getFileUri()}.
     * @return <pre>true</pre> iff the {@link java.net.URI} has a valid scheme specific part,
     * i.e. this {@link com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver} would be able to resolve it.
     */
    boolean validateCorrectness(SourceArtifact artifact);
}
