/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.server.api.upgrade;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.lang.Integer.parseInt;

/**
 * Represents a version of data in repository. Consists of the component, product version and data model version.
 */
public final class Version implements Comparable<Version> {

    public static final String VERSION_0 = "0.0.0";
    public static final List<String> RELEASE_CANDIDATE_CLASSIFIERS =
            Collections.unmodifiableList(Arrays.asList("alpha", "beta", "rc"));

    private String component;
    private int major;
    private int minor;
    private int micro;
    private String classifier = "";
    private int dataModel;

    public static Version valueOf(String component, String versionString) {
        Matcher matcher = Pattern.compile("([0-9]+)\\.([0-9]+)\\.?([0-9]+)?-?([^#]*)(#([0-9]+))?$").matcher(versionString);
        if(!matcher.matches()) {
            throw new RuntimeException("Unsupported version number format " + versionString);
        }

        Version version = new Version();
        version.component = component;
        version.major = parseInt(matcher.group(1));
        version.minor = parseInt(matcher.group(2));

        if (matcher.group(3) != null) {
            version.micro = parseInt(matcher.group(3));
        }
        if (matcher.group(4) != null) {
            version.classifier = matcher.group(4);
        }
        if (matcher.group(5) != null) {
            version.dataModel = parseInt(matcher.group(6));
        }

        return version;
    }

    public String getComponent() {
        return component;
    }

    public int getMajor() {
        return major;
    }

    public int getMinor() {
        return minor;
    }

    public int getMicro() {
        return micro;
    }

    public String getClassifier() {
        return classifier;
    }

    public int getDataModel() {
        return dataModel;
    }

    public String getVersion() {
        StringBuilder builder = new StringBuilder(String.format("%d.%d.%d", major, minor, micro));
        if (!classifier.isEmpty()) {
            builder.append(String.format("-%s", classifier));
        }

        if (dataModel != 0) {
            builder.append(String.format("#%d", dataModel));
        }

        return builder.toString();
    }

    @Override
    public String toString() {
        return String.format("%s %s", component, getVersion());
    }

    @Override
    public int compareTo(Version o) {
        if (!component.equals(o.component)) {
            throw new IllegalArgumentException("Cannot compare versions for 2 components: " + toString() + " and " + o.toString());
        }

        if (major != o.major) {
            return major - o.major;
        } else if (minor != o.minor) {
            return minor - o.minor;
        } else if (micro != o.micro) {
            return micro - o.micro;
        } else if (dataModel != o.dataModel) {
            return dataModel - o.dataModel;
        } else if (isReleaseCandidateClassifier(classifier) && !isReleaseCandidateClassifier(o.classifier)) {
            return -1;
        } else if (!isReleaseCandidateClassifier(classifier) && isReleaseCandidateClassifier(o.classifier)) {
            return 1;
        }
        return classifier.compareTo(o.classifier);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Version version = (Version) o;

        return major == version.major
                && micro == version.micro
                && minor == version.minor
                && classifier.equals(version.classifier)
                && component.equals(version.component)
                && dataModel == version.dataModel;

    }

    @Override
    public int hashCode() {
        int result = component.hashCode();
        result = 31 * result + major;
        result = 31 * result + minor;
        result = 31 * result + micro;
        result = 31 * result + classifier.hashCode();
        result = 31 * result + dataModel;
        return result;
    }

    public boolean isVersion0() {
        return this.equals(Version.valueOf(component, VERSION_0));
    }

    private boolean isReleaseCandidateClassifier(String classifier) {
        for (String releaseCandidateClassifier : RELEASE_CANDIDATE_CLASSIFIERS) {
            if (classifier.startsWith(releaseCandidateClassifier)) {
                return true;
            }
        }
        return false;
    }
}
