package com.xebialabs.deployit.engine.api;

import java.util.List;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

/**
 * Manages the roles in Deployit's security system.
 */
@Path("/security/role")
@Consumes(MediaType.APPLICATION_XML)
@Produces(MediaType.APPLICATION_XML)
public interface RoleService {

    /**
     * Lists the names of all available roles in the security system.
     *
     * @return a list of role names.
     */
    @GET
    List<String> list();

    /**
     * Lists the roles of the currently logged in user.
     *
     * @return a list of role names.
     */
    @GET
    @Path("roles")
    List<String> listMyRoles();

    /**
     * Lists the roles of a user.
     *
     * @permission security#edit
     * @param username the username of the principal
     * @return a list of role names.
     */
    @GET
    @Path("roles/{username}")
    List<String> listRoles(String username);

    /**
     * Creates a new role. Does nothing if the role already exists.
     *
     * @permission security#edit
     * @param name the name of the new role.
     */
    @PUT
    @Path("{role}")
    void create(@PathParam("role") String name);

    /**
     * Assigns a role to a user or group. The role will be created if it doesn't exist yet.
     *
     * @permission security#edit
     * @param name the name of the role to assign
     * @param principal the user or group to assign the role to.
     */
    @PUT
    @Path("{role}/{principal}")
    void assign(@PathParam("role") String name, @PathParam("principal") String principal);

    /**
     * Removes a role from a user or group.
     *
     * @permission security#edit
     * @param name the name of the role to remove
     * @param principal the user or group to remove the role from.
     */
    @DELETE
    @Path("{role}/{principal}")
    void unassign(@PathParam("role") String name, @PathParam("principal") String principal);

    /**
     * Renames a role. Does nothing if the role does not exist.
     *
     * @permission security#edit
     * @param name the current name
     * @param newName the new name
     */
    @POST
    @Path("{role}")
    void rename(@PathParam("role") String name, String newName);

    /**
     * Removes a role from the Deployit security system. All assignments to principals are deleted as well.
     *
     * @permission security#edit
     * @param name the name of the role to delete.
     */
    @DELETE
    @Path("{role}")
    void delete(@PathParam("role") String name);
}
