package com.xebialabs.deployit.engine.api;

import java.util.Collection;
import java.util.Map;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

/**
 * Manages permissions in Deployit.
 */
@Path("/security")
@Consumes({ MediaType.APPLICATION_XML })
@Produces({ MediaType.APPLICATION_XML })
public interface PermissionService {

    /**
     * Gets all the permissions granted to a role.
     *
     * @permission security#edit
     * @param role the role to get the permissions for.
     * @return a Map of configuration item ids to permissions granted to the user.
     */
    @GET
    @Path("granted-permissions/{role}")
    Map<String, Collection<String>> getGrantedPermissions(@PathParam("role") String role);

    /**
     * Gets all the permissions granted to the logged in user.
     *
     * @return a Map of configuration item ids to permissions granted the currently logged in user.
     */
    @GET
    @Path("granted-permissions")
    Map<String, Collection<String>> getMyGrantedPermissions();

    /**
     * Checks if a permission is granted to a role on a CI.
     *
     * @permission security#edit
     * @param permission the name of the permission to check.
     * @param id the path of the CI to check the permission on.
     * @param role the role to which the permission should be granted.
     * @return {@code true} if granted.
     */
    @GET
    @Path("permission/{permission}/{role}/{id:.*?}")
    boolean isGranted(@PathParam("permission") String permission, @PathParam("id") String id, @PathParam("role") String role);

    /**
     * Grants a permission to a role on a CI.
     *
     * @permission security#edit
     * @param permission the name of the permission to grant.
     * @param id the path of the CI to grant the permission on.
     * @param role the role to which the permission should be granted.
     */
    @PUT
    @Path("permission/{permission}/{role}/{id:.*?}")
    void grant(@PathParam("permission") String permission, @PathParam("id") String id, @PathParam("role") String role);

    /**
     * Revokes the permission of a role on a CI.
     *
     *
     * @permission security#edit
     * @param permission the name of the permission to revoke.
     * @param id the path of the CI to revoke the permission from.
     * @param role the role from which the permission should be revoked.
     */
    @DELETE
    @Path("permission/{permission}/{role}/{id:.*?}")
    void revoke(@PathParam("permission") String permission, @PathParam("id") String id, @PathParam("role") String role);

    /**
     * <i><b>Deprecated - </b>Use {@link #isGrantedToMe(String, String) GET /security/check}</i>.
     * Checks if the currently logged in user has a certain permission on a CI. If the user does not have this permission, error code {@code 403} is returned.
     *
     * @param permission the name of the permission to check.
     * @param id the path of the CI to check the permission on.
     * @deprecated Use {@link #isGrantedToMe(String, String) GET /security/check}.
     */
    @Deprecated
    @HEAD
    @Path("check/{permission}/{id:.*?}")
    void checkMyPermission(@PathParam("permission") String permission, @PathParam("id") String id);

    /**
     * Checks if the currently logged in user has a certain permission on a CI.
     *
     * @param permission the name of the permission to check.
     * @param id the path of the CI to check the permission on.
     * @return {@code true} if the user has this permission; {@code false} otherwise.
     */
    @GET
    @Path("check/{permission}/{id:.*?}")
    boolean isGrantedToMe(@PathParam("permission") String permission, @PathParam("id") String id);
}
