package com.xebialabs.deployit.engine.api;

import java.util.List;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.plugin.api.reflect.Type;

/**
 * Deployment management
 */
@Path("/deployment")
@Consumes(MediaType.APPLICATION_XML)
@Produces(MediaType.APPLICATION_XML)
public interface DeploymentService {

    /**
     * Prepares an initial deployment.
     *
     * @permission deploy#initial
     * @param versionId     The ID of the {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} that is the source of the deployment.
     * @param environmentId The ID of the {@link com.xebialabs.deployit.plugin.api.udm.Environment udm.Environment} that is the target of the deployment.
     * @return A new <code>Deployment</code> object to which you can add deployeds.
     */
    @GET
    @Path("prepare/initial")
    Deployment prepareInitial(@QueryParam("version") String versionId, @QueryParam("environment") String environmentId);

    /**
     * Prepares an update deployment.
     *
     * @permission deploy#upgrade
     * @param versionId             The ID of the new {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} that is the source of the deployment.
     * @param deployedApplicationId The ID of the {@link com.xebialabs.deployit.plugin.api.udm.DeployedApplication udm.DeployedApplication} that is to be updated.
     * @return A new <code>Deployment</code> object which contains the updated deployeds.
     */
    @GET
    @Path("prepare/update")
    Deployment prepareUpdate(@QueryParam("version") String versionId, @QueryParam("deployedApplication") String deployedApplicationId);

    /**
     * Prepares an undeployment.
     *
     * @permission deploy#undeploy
     * @param deployedApplicationId The ID of the {@link com.xebialabs.deployit.plugin.api.udm.DeployedApplication udm.DeployedApplication} that is to be undeployed.
     * @return A new <code>Deployment</code> object which contains no deployeds (ie. all deployeds of the previous deployment will be deleted).
     */
    @GET
    @Path("prepare/undeploy")
    Deployment prepareUndeploy(@QueryParam("deployedApplication") String deployedApplicationId);

    /**
     * Generates all deployeds for the given deployment.
     *
     * @permission deploy#initial
     * @param deployment The prepared Deployment parameter object.
     * @return An updated Deployment parameter object.
     */
    @POST
    @Path("generate/all")
    Deployment generateAllDeployeds(Deployment deployment);

    /**
     * Generates only the deployeds for the given deployables in the deployment.
     * <p/>
     * Try to generate deployeds from each of the deployables to all the matching {@link com.xebialabs.deployit.plugin.api.udm.Container Containers}
     * in the {@link com.xebialabs.deployit.plugin.api.udm.Environment Environment} of the Deployment parameter object.
     * <p/>
     *
     * @permission deploy#initial
     * @param deployableIds The list of IDs of deployables to generate the deployeds for.
     * @param deployment    The prepared Deployment parameter object
     * @return An updated Deployment parameter object.
     */
    @POST
    @Path("generate/selected")
    Deployment generateSelectedDeployeds(@QueryParam("deployables") List<String> deployableIds, Deployment deployment);

    /**
     * Generates a deployed for the given deployable to the given container in the deployment.
     * <p/>
     * The deployed should be of the given type (which is optional)
     * <p/>
     *
     * @permission deploy#initial
     * @param deployableId The ID of the deployable to generate a deployed for
     * @param containerId  The ID of the container to generate a deployed to
     * @param deployedType (Optional) The type of the deployed to generate
     * @param deployment    The prepared Deployment parameter object
     * @return An updated Deployment parameter object.
     */
    @POST
    @Path("generate/single")
    Deployment generateSingleDeployed(@QueryParam("deployable") String deployableId, @QueryParam("container") String containerId, @QueryParam("deployedtype") Type deployedType, Deployment deployment);

    /**
     * Validates the generated deployment.
     *
     * Checks whether all the deployeds that are in the deployment are valid.
     *
     * @param deployment The deployment to validate.
     * @return The validated deployment.
     */
    @POST
    @Path("validate")
    Deployment validate(Deployment deployment);

    /**
     * Creates the deployment task.
     *
     * @permission deploy#initial for initial deployments
     * @permission deploy#upgrade for upgrades
     * @permission deploy#remove for undeployments
     * @param deployment The fully prepared Deployment parameter object.
     * @return a reference to a Task ID that can be executed by the {@link TaskService}.
     */
    @POST
    String createTask(Deployment deployment);

    /**
     * Rollback a {@code STOPPED} or {@code EXECUTED} task. Reverting the deployment to the previous state. The task
     * will be set to {@code CANCELLED} when it was {@code STOPPED}, and {@code DONE} when it was {@code EXECUTED}.
     *
     * @param taskId the ID of the task
     * @return the ID of the new task.
     */
    @POST
    @Path("rollback/{taskid}")
    String rollback(@PathParam("taskid") String taskId);

}

