package com.xebialabs.deployit.engine.api;

import java.util.List;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

import org.jboss.resteasy.annotations.providers.multipart.MultipartForm;
import org.joda.time.DateTime;

import com.xebialabs.deployit.engine.api.dto.ArtifactAndData;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

/**
 * Provides access to the Deployit repository.
 */
@Path("/repository")
@Consumes({ MediaType.APPLICATION_XML })
@Produces({ MediaType.APPLICATION_XML })
public interface RepositoryService {

    /**
     * Reads a configuration item from the repository.
     *
     * @permission read
     * @param ID
     *            the ID of the CI
     * @return the CI, or a {@code 404} error code if not found.
     */
    @GET
    @Path("ci/{ID:.*?}")
    ConfigurationItem read(@PathParam("ID") String ID);

    /**
     * Creates a new configuration item.
     *
     * @permission repo#edit
     * @param ID
     *            the ID/path of the new CI.
     * @param configurationItem
     *            the data of the CI.
     * @return the newly created CI
     */
    @POST
    @Path("ci/{ID:.*?}")
    ConfigurationItem create(@PathParam("ID") String ID, ConfigurationItem configurationItem);

    /**
     * Creates a new artifact CI with data.
     *
     * @permission repo#edit
     * @param ID
     *            the ID/path of the new artifact.
     * @param artifactAndData
     *            the data of the artifact.
     * @return the newly created artifact
     */
    @POST
    @Path("ci/{ID:.*?}")
    @Consumes({ MediaType.MULTIPART_FORM_DATA })
    ConfigurationItem create(@PathParam("ID") String ID, @MultipartForm ArtifactAndData artifactAndData);

    /**
     * Modifies a configuration item.
     *
     * @permission repo#edit
     * @param ID
     *            the ID of the CI to modify.
     * @param configurationItem
     *            the new data of the CI.
     * @return the updated CI, or a {@code 409} error code if the CI has been updated on the server and you are sending modifications on an old version.
     */
    @PUT
    @Path("ci/{ID:.*?}")
    ConfigurationItem update(@PathParam("ID") String ID, ConfigurationItem configurationItem);

    /**
     * Modifies an artifact (upload new data).
     *
     * @permission repo#edit
     * @param ID
     *            the ID of the artifact to modify.
     * @param artifactAndData
     *            the new data of the artifact.
     * @return the updated artifact, or a {@code 409} error code if the CI has been updated on the server and you are sending modifications on an old version.
     */
    @PUT
    @Path("ci/{ID:.*?}")
    @Consumes({ MediaType.MULTIPART_FORM_DATA })
    ConfigurationItem update(@PathParam("ID") String ID, @MultipartForm ArtifactAndData artifactAndData);

    /**
     * Deletes a configuration item.
     *
     * @permission repo#edit
     * @permission import#remove
     * @param ID
     *            the ID of the CI to delete.
     */
    @DELETE
    @Path("ci/{ID:.*?}")
    void delete(@PathParam("ID") String ID);

    /**
     * Reads multiple configuration items from the repository.
     *
     * @permission read
     * @param ids
     *            the ids of the CIs
     * @return the CIs, or a {@code 404} error code if one of them was not found.
     */
    @POST
    @Path("cis/read")
    List<ConfigurationItem> read(List<String> ids);

    /**
     * Creates multiple configuration items.
     *
     * @permission repo#edit
     * @param configurationItems
     *            the data of the CIs.
     * @return the newly created CIs
     */
    @POST
    @Path("cis")
    List<ConfigurationItem> create(List<ConfigurationItem> configurationItems);

    /**
     * Modifies multiple configuration items.
     *
     * @permission repo#edit
     * @param configurationItems
     *            the new data of the CIs.
     * @return the updated CIs, or a {@code 409} error code if one of the CIs has been updated on the server and you are sending modifications on an old version.

     */
    @PUT
    @Path("cis")
    List<ConfigurationItem> update(List<ConfigurationItem> configurationItems);

    /**
     * Checks if a configuration item exists.
     *
     * @param ID
     *            the ID of the CI to check
     * @return {@code true} if the CI exists, {@code false} if not.
     */
    @GET
    @Path("exists/{ID:.*?}")
    Boolean exists(@PathParam("ID") String ID);

    /**
     * Moves a configuration item in the repository.
     *
     * @permission repo#edit
     * @param ID
     *            the ID of the CI to move
     * @param newLocation
     *            the new ID of the CI, reflecting the path in the repository.
     * @return the moved CI
     */
    @POST
    @Path("move/{ID:.*?}")
    ConfigurationItem move(@PathParam("ID") String ID, @QueryParam("newId") String newLocation);

    /**
     * Changes the name of a configuration item in the repository.
     *
     * @permission repo#edit
     * @param ID
     *            the ID of the CI to rename
     * @param newName
     *            the new name.
     * @return the renamed CI
     */
    @POST
    @Path("rename/{ID:.*?}")
    ConfigurationItem rename(@PathParam("ID") String ID, @QueryParam("newName") String newName);

    /**
     * Retrieves configuration items by way of a query. All parameters are
     * optional.
     *
     * @param type
     *            the type of the CI
     * @param parent
     *            the parent ID of the CI. If set, only the subtree of this CI
     *            is searched.
     * @param namePattern
     *            a search pattern for the name. This is like the SQL "LIKE"
     *            pattern: the character '%' represents any string of zero or
     *            more characters, and the character '_' (underscore) represents
     *            any single character. Any literal use of these two characters
     *            must be escaped with a backslash ('\'). Consequently, any
     *            literal instance of a backslash must also be escaped,
     *            resulting in a double backslash ('\\').
     * @param lastModifiedBefore look for CIs modified before this date.
     * @param lastModifiedAfter look for CIs modified after this date.
     * @param page the desired page, in case of a paged query.
     * @param resultPerPage the page size, or {@code -1} for no paging.
     * @return a list of references to CIs
     */
    @GET
    @Path("query")
    List<ConfigurationItemId> query(
            @QueryParam("type") Type type,
            @QueryParam("parent") String parent,
            @QueryParam("namePattern") String namePattern,
            @QueryParam("lastModifiedBefore") DateTime lastModifiedBefore,
            @QueryParam("lastModifiedAfter") DateTime lastModifiedAfter,
            @QueryParam("page") long page,
            @QueryParam("resultsPerPage") long resultPerPage);

    /**
     * Validate the configuration items, returning any validation errors found.
     *
     * @param cis The list of configuration items to valdiate.
     *
     * @return The validated configuration items, including any validation messages.
     */
    @POST
    @Path("validate")
    List<ConfigurationItem> validate(List<ConfigurationItem> cis);
}
