/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api;

import java.util.List;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

import com.xebialabs.deployit.engine.api.dto.ServerInfo;
import com.xebialabs.deployit.engine.api.dto.ServerState;
import com.xebialabs.xlplatform.documentation.PublicApi;

/**
 * Services related to the operation of the XL Deploy server process.
 */
@Path("/server")
@Produces({MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
@PublicApi
public interface ServerService {

    /**
     * Returns information about the configuration of the sever. For example: version, plugins, and classpath.
     *
     * @return an information object.
     */
    @GET
    @Path("info")
    ServerInfo getInfo();

    /**
     * Runs the garbage collector on the repository.
     *
     * @permission admin
     */
    @POST
    @Path("gc")
    void gc();

    /**
     * Reload and validate the license file from disk.
     *
     * @permission admin
     */
    @POST
    @Path("license/reload")
    void licenseReload();

    /**
     * Logs out the current user.
     */
    @POST
    @Path("logout")
    void logout();

    /**
     * View as user.
     *
     * @permission admin
     */
    @PUT
    @Path("view-as")
    void viewAs(@QueryParam("user") String username, @QueryParam("role") List<String> roles);

    /**
     * Stops the server process in a graceful manner.
     * @apiNote shutdown can take anywhere from one to a few minutes in order to
     * gracefully stop all running parts of the server
     *
     * @permission admin
     * @return PID of the JVM process
     */
    @POST
    @Path("shutdown")
    @Produces({MediaType.TEXT_PLAIN})
    Long shutdown();


    /**
     * Put server into MAINTENANCE mode (prepare for shutdown).
     *
     * @permission admin
     */
    @POST
    @Path("maintenance/start")
    void startMaintenance();

    /**
     * Put server into RUNNING mode.
     *
     * @permission admin
     */
    @POST
    @Path("maintenance/stop")
    void stopMaintenance();


    /**
     * Returns the information about the current server state (is it RUNNING or in MAINTENANCE mode).
     *
     * @return server state (RUNNING, MAINTENANCE)
     */
    @GET
    @Path("state")
    ServerState getState();
}
