/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api;

import com.xebialabs.deployit.engine.api.dto.TaskPathStatus;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.deployit.engine.api.execution.*;
import com.xebialabs.xlplatform.documentation.PublicApi;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import java.util.List;
import java.util.stream.Stream;

/**
 * Manages tasks with blocks on the XL Deploy Server.
 */
@Path("/tasks/v2")
@Consumes({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@Produces({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@PublicApi
public interface TaskBlockService {

    /**
     * Returns the active tasks of the logged in user.
     * By default returns full tasks, but summary mode should be preferred for performance.
     *
     * @param fetchMode whether to strip block data for tasks or to do a full fetch
     * @return a list of tasks with blocks.
     */
    @GET
    @Path("current")
    List<TaskWithBlock> getMyCurrentTasks(@QueryParam("fetchMode") FetchMode fetchMode);

    /**
     * Returns all active tasks for all users. Only allowed for admin.
     * By default returns full tasks, but summary mode should be preferred for performance.
     *
     * @param fetchMode whether to strip block data for tasks or to do a full fetch
     * @return a list of tasks with blocks.
     */
    @GET
    @Path("current/all")
    List<TaskWithBlock> getAllCurrentTasks(@QueryParam("fetchMode") FetchMode fetchMode);

    /**
     * Returns all active satellite tasks the logged in user.
     * By default returns full tasks, but summary mode should be preferred for performance.
     *
     * @param satelliteId the ID of the satellite where task will be executed
     * @param fetchMode whether to strip block data for tasks or to do a full fetch
     * @return a list of tasks with blocks.
     */
    @GET
    @Path("current/satellite")
    List<TaskWithBlock> getMySatelliteTasks(@QueryParam("satelliteId") String satelliteId,
                                            @QueryParam("fetchMode") FetchMode fetchMode);

    /**
     * Returns all active satellite tasks for all users. Only allowed for admin.
     * By default returns full tasks, but summary mode should be preferred for performance.
     *
     * @param satelliteId the ID of the satellite where task will be executed
     * @param fetchMode whether to strip block data for tasks or to do a full fetch
     * @return a list of tasks with blocks.
     */
    @GET
    @Path("current/all/satellite")
    List<TaskWithBlock> getCurrentSatelliteTasks(@QueryParam("satelliteId") String satelliteId,
                                                 @QueryParam("fetchMode") FetchMode fetchMode);

    /**
     * Returns a task by ID.
     *
     * @permission Available only to admin and owner of the task
     * @param taskId the ID of the task
     * @return the task with blocks.
     */
    @GET
    @Path("{taskid}")
    TaskWithBlock getTask(@PathParam("taskid") String taskId);

    /**
     * Returns a block by ID.
     *
     * @permission Available only to admin and owner of the task
     * @param taskId the ID of the task
     * @param blockPath the ID of the block
     * @return the block.
     */
    @GET
    @Path("{taskid}/block/{blockPath}")
    BlockState getBlock(@PathParam("taskid") String taskId, @PathParam("blockPath") String blockPath);

    /**
     * Returns a block with steps by ID.
     *
     * @permission Available only to admin and owner of the task
     * @param taskId the ID of the task
     * @param blockPath the ID of the block
     * @return the block with steps.
     */
    @GET
    @Path("{taskid}/block/{blockPath}/step")
    StepBlockState getSteps(@PathParam("taskid") String taskId, @PathParam("blockPath") String blockPath);

    /**
     * Retrieves information about a step.
     * @restDetails The call returns a response with status code of 304 (Not Modified) if the step has not been modified since {@code ifModifiedSince},
     * otherwise returns a status code of 200 (OK) and the XML containing a StepState.
     *
     * @apiDetails The method throws a {@link java.lang.RuntimeException} if the step has not been modified since {@code ifModifiedSince}, otherwise
     * returns a {@link com.xebialabs.deployit.engine.api.execution.StepState StepState}
     *
     * @permission Available only to admin and owner of the task
     * @param taskId          the ID of the task
     * @param stepPath        the path of the step in the block tree.
     * @param ifModifiedSince the if-modified-since date in <a href="http://datatracker.ietf.org/doc/rfc1123/">RFC 1123</a> (<a href="http://datatracker.ietf.org/doc/rfc822/">RFC 822</a> with 4-digit years) date format.
     * @return XML containing a StepState if the step has not been modified since {@code ifModifiedSince}
     */
    @GET
    @Path("{taskid}/step/{stepPath}")
    StepState getStep(@PathParam("taskid") String taskId, @PathParam("stepPath") String stepPath, @HeaderParam("If-Modified-Since") DateTime ifModifiedSince);

    /**
     * Starts a task.
     *
     * @permission Available only to admin and owner of the task
     * @param taskId the ID of the task
     */
    @POST
    @Path("{taskid}/start")
    void start(@PathParam("taskid") String taskId);

    /**
     * Schedules a task.
     *
     * @permission Available only to admin and owner of the task
     * @param taskId the ID of the task
     * @param time the local time (including the local timezone) when the task should be started by the server.
     */
    @POST
    @Path("{taskid}/schedule")
    void schedule(@PathParam("taskid") String taskId, @QueryParam("time") DateTime time);

    /**
     * Gracefully stops an active task.
     *
     * @permission Available only to admin and owner of the task
     * @param taskId the ID of the task
     */
    @POST
    @Path("{taskid}/stop")
    void stop(@PathParam("taskid") String taskId);

    /**
     * Aborts an active task.
     *
     * @permission Available only to admin and owner of the task
     * @param taskId the ID of the task
     */
    @POST
    @Path("{taskid}/abort")
    void abort(@PathParam("taskid") String taskId);

    /**
     * Cancels a stopped task.
     *
     * @permission Available only to admin and owner of the task
     * @param taskId the ID of the task
     */
    @DELETE
    @Path("{taskid}")
    void cancel(@PathParam("taskid") String taskId);

    /**
     * Force-cancels a stopped task - do a best-effort cleanup, continuing on when steps of an always-run phase fail
     *
     * @permission Available only to admin
     * @param taskId the ID of the task
     */
    @DELETE
    @Path("{taskid}/force")
    void forceCancel(@PathParam("taskid") String taskId);

    /**
     * Force-cancels stopped tasks - do a best-effort cleanup, continuing on when steps of an always-run phase fail
     *
     * @permission Available only to admin
     * @param taskIds the IDs of the tasks
     */
    @POST
    @Path("/force-cancel")
    void forceCancelTasks(List<String> taskIds);

    /**
     * Archive an executed task.
     *
     * @permission Available only to admin and owner of the task
     * @param taskId the ID of the task
     */
    @POST
    @Path("{taskid}/archive")
    void archive(@PathParam("taskid") String taskId);

    /**
     * Indicates that one or more steps should be skipped.
     *
     * @permission task#skip_step
     * @param taskId the ID of the task
     * @param stepIds the paths of the steps to skip
     * @return a task
     */
    @POST
    @Path("{taskid}/skip")
    TaskWithBlock skip(@PathParam("taskid") String taskId, List<String> stepIds);

    /**
     * Indicates that one or more steps should no longer be skipped, but executed.
     *
     * @permission task#skip_step
     * @param taskId the ID of the task
     * @param stepIds the paths of the steps to unskip
     * @return a task
     */
    @POST
    @Path("{taskid}/unskip")
    TaskWithBlock unskip(@PathParam("taskid") String taskId, List<String> stepIds);

    /**
     * Add a pause step at the specified position.
     *
     * @permission Available only to admin and owner of the task
     * @param taskId the ID of the task
     * @param stepPath the path of the step
     * @return a task with full step information.
     */
    @POST
    @Path("{taskid}/add-pause/{stepPath}")
    StepBlockState addPause(@PathParam("taskid") String taskId, @PathParam("stepPath") String stepPath);

    /**
     * Assigns a task to a different user.
     *
     * @permission task#assign
     * @param taskId the ID of the task
     * @param owner the name of the user that will be the new owner of the task.
     * @return the task.
     */
    @POST
    @Path("{taskid}/assign/{owner}")
    TaskWithBlock assign(@PathParam("taskid") String taskId, @PathParam("owner") String owner);

    /**
     * Takeover a task from the owner. This will re-assign the task to your logged in user.
     * This is the reverse of the {@link #assign(String, String)} method, which allows you to re-assign
     * a task.
     *
     * @permission task#takeover
     * @param taskId the ID of the task to take over
     * @param owner the current owner of the task
     * @return the task you took over.
     */
    @POST
    @Path("{taskid}/takeover/{owner}")
    TaskWithBlock takeover(@PathParam("taskid") String taskId, @PathParam("owner") String owner);

    /**
     * Searches for archived tasks without step information.
     *
     * @permission report#view
     * @param begin The first day we want to see tasks from, or {@code null} for no such limit.
     * @param end The last day we want to see tasks from, or {@code null} for no such limit.
     * @param paging {@link com.xebialabs.deployit.engine.api.dto.Paging Paging} paging parameters to fetch tasks page by page.
     * @return a list of tasks limited to maximum of value of {@code deploy.server.rest.api.maxPageSize} per page. response header X-Total-Count determines the total number of tasks.
     */
    @GET
    @Path("query")
    Stream<TaskWithBlock> query(@QueryParam("begindate") LocalDate begin, @QueryParam("enddate") LocalDate end, @BeanParam Paging paging);

    /**
     * Searches for tasks with detailed step information.
     *
     * @permission report#view
     * @param begin The first day we want to see tasks from, or {@code null} for no such limit.
     * @param end The last day we want to see tasks from, or {@code null} for no such limit.
     * @param paging {@link com.xebialabs.deployit.engine.api.dto.Paging Paging} paging parameters to fetch tasks page by page.
     * @return a list of tasks limited to maximum of value of {@code deploy.server.rest.api.maxPageSize} per page. response header X-Total-Count determines the total number of tasks.
     */
    @GET
    @Path("export")
    @IncludeStepBlocks
    Stream<TaskWithBlock> export(@QueryParam("begindate") LocalDate begin, @QueryParam("enddate") LocalDate end, @BeanParam Paging paging);

    /**
     * Returns task status by path.
     *
     * @param taskId the ID of the task
     * @param path   path
     * @return the block.
     */
    @GET
    @Path("{taskid}/status/{path}")
    TaskPathStatus getStatus(@PathParam("taskid") String taskId, @PathParam("path") String path);


}
