/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api;

import com.xebialabs.deployit.engine.api.dto.FileUpload;
import com.xebialabs.deployit.engine.api.dto.ImportFromUrlParams;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApi;

import org.jboss.resteasy.annotations.providers.multipart.MultipartForm;

import java.util.List;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;

/**
 * Manages Deployment Packages.
 */
@Path("/package")
@Consumes({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@Produces({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@PublicApi
public interface PackageService {

    /**
     * Lists all packages that are present in the {@code importablePackages} directory on the XL Deploy Server.
     *
     * @return List of names of packages that can be imported.
     */
    @GET
    @Path("import")
    List<String> list();

    /**
     * Imports a package that is present in the {@code importablePackages} directory on the XL Deploy Server.
     *
     * @permission import#initial for importing the first version of a package
     * @permission import#upgrade if another version of the package already exists
     * @param file The name of the package to import
     * @return The {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} CI that is created after importing.
     */
    @POST
    @Path("import/{file:.+}")
    ConfigurationItem importPackage(@PathParam("file") String file);

    /**
     * Fetches a package from a URL outside of the XL Deploy Server.
     *
     * @permission import#initial for importing the first version of a package
     * @permission import#upgrade if another version of the package already exists
     * @param url The URL to import from
     * @return The {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} CI that is created after importing.
     * @deprecated Use {@link PackageService#fetch(ImportFromUrlParams)}
     */
    @Deprecated
    @POST
    @Path("fetch")
    ConfigurationItem fetch(String url);

    /**
     * Fetches a package from a URL outside of the XL Deploy Server.
     * Supports authentication
     *
     * @permission import#initial for importing the first version of a package
     * @permission import#upgrade if another version of the package already exists
     * @param params The import URL parameters
     * @return The {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} CI that is created after importing.
     */
    @POST
    @Path("fetch2")
    ConfigurationItem fetch(ImportFromUrlParams params);

    /**
     * Uploads a package to the XL Deploy Server.
     *
     * @permission import#initial for importing the first version of a package
     * @permission import#upgrade if another version of the package already exists
     * @param file The name of the package to be imported
     * @param form The upload form.
     * @return The {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} CI that is created after importing.
     */
    @POST
    @Path("upload/{file:.+}")
    @Consumes(MediaType.MULTIPART_FORM_DATA)
    ConfigurationItem upload(@PathParam("file") String file, @MultipartForm FileUpload form);
}
