/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.xlplatform.security.dto.PasswordValidationResult;
import com.xebialabs.deployit.engine.api.security.User;
import com.xebialabs.xlplatform.documentation.PublicApi;

import java.util.List;

/**
 * Manages users in XL Deploy's internal user repository. Use this service only if you're not connected to an external user repository like LDAP.
 */
@Path("/security/user")
@Consumes({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@Produces({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@PublicApi
public interface UserService {

    /**
     * Creates a new user.
     *
     * @permission security#edit
     * @param username the login name
     * @param user the details of the user, containing username and new password. Username in here should be the same as the username in the URL path.
     * @return the created user, without the password.
     */
    @POST
    @Path("{username}")
    User create(@PathParam("username") String username, User user);

    /**
     * Returns the user details without the password.
     *
     * @permission security#edit
     * @param username the login name of the user.
     * @return the details of the user.
     */
    @GET
    @Path("{username}")
    User read(@PathParam("username") String username);

    /**
     * Returns the user details without the password.
     *
     * @permission security#edit
     * @return the details of the user.
     */
    @GET
    List<String> listUserNames(
            @QueryParam("username") String username,
            @BeanParam Paging paging,
            @QueryParam("order") Ordering order);

    /**
     * Changes the password of the indicated user.
     * @restDetails Username should be the same as the username in the URL path.
     *
     * @permission security#edit
     * @param username the login name of the user.
     * @param user the details of the user, containing username and new password.
     */
    @PUT
    @Path("{username}")
    void modifyPassword(@PathParam("username") String username, User user);

    @POST
    @Path("validatePassword")
    List<PasswordValidationResult> validatePassword(User user);

    /**
     * Deletes the user.
     *
     * @permission security#edit
     * @param username the login name of the user.
     */
    @DELETE
    @Path("{username}")
    void delete(@PathParam("username") String username);
}
