package com.xebialabs.deployit.documentation;

import static java.lang.String.format;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.*;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.ConfigurationItem;
import org.reflections.Reflections;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyType;
import org.reflections.scanners.TypeAnnotationsScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Generates markdown documentation for CIs found on the classpath.
 */
public class PluginCIMarkdownGenerator {
	public static void main(String[] args) throws SecurityException, NoSuchFieldException, IOException {
		String filter = "com.xebialabs.deployit.ci";
		String outputFilename = null;
		
		if (args.length > 0) {
			outputFilename = args[0];
		}

		if (args.length > 1) {
			filter = args[1];
		}

		logger.info("Generating documentation for CIs matching " + filter + " and appending to file " + outputFilename);

		final Reflections reflections = new Reflections(filter, new TypeAnnotationsScanner());
		final Set<Class<?>> annotatedWith = reflections.getTypesAnnotatedWith(ConfigurationItem.class);
		List<ConfigurationItemDescriptor> toPrint = new ArrayList<ConfigurationItemDescriptor>();
		for (Class<?> aClass : annotatedWith) {
			logger.info("Adding class {}", aClass);
			toPrint.add(new ConfigurationItemDescriptor(aClass, Maps.<String, String>newHashMap()));
		}

		Collections.sort(toPrint, new Comparator<ConfigurationItemDescriptor>() {
			@Override
			public int compare(ConfigurationItemDescriptor o1, ConfigurationItemDescriptor o2) {
				return o1.getSimpleName().compareTo(o2.getSimpleName());
			}
		});
		
		StringBuilder sb = new StringBuilder();
		for (ConfigurationItemDescriptor configurationItemDescriptor : toPrint) {
			printHelp(sb, configurationItemDescriptor);
		}

		appendToFile(outputFilename, sb);
	}

	private static void appendToFile(String outputFilename, StringBuilder sb) throws IOException {
		FileWriter fw = new FileWriter(new File(outputFilename), true);
		fw.append(sb);
		fw.close();
	}

	private static void printHelp(StringBuilder sb, ConfigurationItemDescriptor ciDescriptor) {
        sb.append("\n");
        Formatter fmt = new Formatter(sb);
        fmt.format("## %s ##\n\n", ciDescriptor.getSimpleName());
        fmt.format("%s\n\n", ciDescriptor.getDescription());
        fmt.format("_Type_: %s\n\n", ciDescriptor.getType());
        
        if (ciDescriptor.getPropertyDescriptors().length > 0) {
        	fmt.format("_Properties:_\n\n");
	        for (ConfigurationItemPropertyDescriptor property : sortProperties(ciDescriptor.getPropertyDescriptors())) {
	        	fmt.format("* %s%s(%s)%s: %s\n", property.isRequired() ? "**" : "_", property.getName(), getType(property), property.isRequired() ? "**" : "_", property.getDescription());
	            if (property.getType() == ConfigurationItemPropertyType.ENUM) {
	            	fmt.format("    * Values: %s\n", Arrays.toString(property.getEnumValues()));
	            }
	        }
        }
        
        sb.append("\n");
        sb.append("\n");
	}
	
    private static Collection<ConfigurationItemPropertyDescriptor> sortProperties(ConfigurationItemPropertyDescriptor[] propertyDescriptors) {
    	List<ConfigurationItemPropertyDescriptor> result = Arrays.asList(propertyDescriptors);
    	Collections.sort(result, new Comparator<ConfigurationItemPropertyDescriptor>() {
    		@Override
    		public int compare(ConfigurationItemPropertyDescriptor o1, ConfigurationItemPropertyDescriptor o2) {
    			int result = o1.isRequired() == o2.isRequired() ? 0 : (o1.isRequired() ? -1 : 1);
    			if (result == 0) {
    				result = o1.getName().compareTo(o2.getName());
    			}
    			return result;
    		}
		});
		return result;
	}

	private static String getType(final ConfigurationItemPropertyDescriptor property) {
        final ConfigurationItemPropertyType type = property.getType();
        if (type == ConfigurationItemPropertyType.CI) {
            return property.getPropertyClassname();
        } else if (type == ConfigurationItemPropertyType.SET_OF_CIS) {
            return format("Set<%s>", property.getCollectionMemberClassname());
        } else if (type == ConfigurationItemPropertyType.SET_OF_STRINGS) {
            return "Set<String>";
        } else if (type == ConfigurationItemPropertyType.LIST_OF_OBJECTS) {
            return format("List<%s>", property.getCollectionMemberClassname());
        } else {
            return property.getType().name();
        }
    }

	private static final Logger logger = LoggerFactory.getLogger(PluginCIMarkdownGenerator.class);
}
