package com.xebialabs.deployit.client;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;

import java.util.*;

import static com.google.common.collect.ImmutableList.copyOf;
import static com.google.common.collect.Iterables.*;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.transformValues;
import static com.google.common.collect.Maps.uniqueIndex;
import static java.util.Collections.sort;

public class Descriptors {

	private final List<Descriptor> descriptors;
	private final ImmutableMap<String, Descriptor> descriptorMap;
	private final Map<String, List<String>> propertiesIndexedByMap;

	public Descriptors(List<Descriptor> descriptors) {
		this.descriptors = newArrayList(descriptors);
		sort(this.descriptors, new Comparator<Descriptor>() {
			@Override
			public int compare(Descriptor o1, Descriptor o2) {
				return o1.getType().toString().compareTo(o2.getType().toString());
			}
		});

		descriptorMap = uniqueIndex(descriptors, new Function<Descriptor, String>() {
			@Override
			public String apply(Descriptor input) {
				return input.getType().toString();
			}
		});

		propertiesIndexedByMap = transformValues(descriptorMap, new Function<Descriptor, List<String>>() {
			@Override
			public List<String> apply(Descriptor input) {
				final ArrayList<String> properties = newArrayList(transform(input.getPropertyDescriptors(), new Function<PropertyDescriptor, String>() {
					@Override
					public String apply(PropertyDescriptor input) {
						return input.getName();
					}
				}));
				sort(properties);
				return properties;
			}
		});
	}

	public List<Descriptor> getDescriptors() {
		return descriptors;
	}

	public Map<String, List<String>> getPropertiesIndexedByMap() {
		return propertiesIndexedByMap;
	}

	public List<Descriptor> getDeployables() {
		return copyOf(filter(descriptors, new Predicate<Descriptor>() {
			@Override
			public boolean apply(Descriptor input) {
				return input.getInterfaces().contains(Type.valueOf("udm.Deployable"));
			}
		}));
	}

	public List<Descriptor> getDeployableArtifacts() {
		return copyOf(filter(getDeployables(), new Predicate<Descriptor>() {
			@Override
			public boolean apply(Descriptor input) {
				return input.getInterfaces().contains(Type.valueOf("udm.Artifact"));
			}
		}));
	}

	public List<Descriptor> getDeployableResources() {
		return copyOf(filter(getDeployables(), new Predicate<Descriptor>() {
			@Override
			public boolean apply(Descriptor input) {
				return !input.getInterfaces().contains(Type.valueOf("udm.Artifact"));
			}
		}));
	}


	public boolean isPropertyNameofType(final String type, final String propertyname) {
		final Descriptor configurationItemDescriptor = descriptorMap.get(type);
		if (configurationItemDescriptor == null) {
			return false;
		}
		return any(configurationItemDescriptor.getPropertyDescriptors(), new Predicate<PropertyDescriptor>() {
			@Override
			public boolean apply(PropertyDescriptor input) {
				return input.getName().equals(propertyname);
			}
		});
	}

	public List<String> getPropertyName(String type) {
		Preconditions.checkNotNull(Strings.emptyToNull(type), "type should not be null or empty");
		Preconditions.checkArgument(descriptorMap.containsKey(type), "type %s not found", type);

		final Descriptor Descriptor = descriptorMap.get(type);
		final Collection<PropertyDescriptor> propertyDescriptors = Descriptor.getPropertyDescriptors();
		return copyOf(transform(propertyDescriptors, new Function<PropertyDescriptor, String>() {
			@Override
			public String apply(PropertyDescriptor input) {
				return input.getName();
			}
		}));
	}
}
