package com.xebialabs.deployit.client;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyDescriptorDto;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.ImmutableList.copyOf;
import static com.google.common.collect.Iterables.*;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.transformValues;
import static com.google.common.collect.Maps.uniqueIndex;
import static java.util.Collections.sort;

public class Descriptors {

	private final List<ConfigurationItemDescriptorDto> descriptors;
	private final ImmutableMap<String, ConfigurationItemDescriptorDto> descriptorMap;
	private final Map<String, List<String>> propertiesIndexedByMap;

	public Descriptors(List<ConfigurationItemDescriptorDto> descriptors) {
		this.descriptors = newArrayList(descriptors);
		sort(this.descriptors, new Comparator<ConfigurationItemDescriptorDto>() {
			@Override
			public int compare(ConfigurationItemDescriptorDto o1, ConfigurationItemDescriptorDto o2) {
				return o1.getType().compareTo(o2.getType());
			}
		});

		descriptorMap = uniqueIndex(descriptors, new Function<ConfigurationItemDescriptorDto, String>() {
			@Override
			public String apply(ConfigurationItemDescriptorDto input) {
				return input.getType();
			}
		});

		propertiesIndexedByMap = transformValues(descriptorMap, new Function<ConfigurationItemDescriptorDto, List<String>>() {
			@Override
			public List<String> apply(@Nullable ConfigurationItemDescriptorDto input) {
				final ArrayList<String> properties = newArrayList(transform(input.getPropertyDescriptors(), new Function<ConfigurationItemPropertyDescriptorDto, String>() {
					@Override
					public String apply(@Nullable ConfigurationItemPropertyDescriptorDto input) {
						return input.getName();
					}
				}));
				sort(properties);
				return properties;
			}
		});
	}

	public List<ConfigurationItemDescriptorDto> getDescriptors() {
		return descriptors;
	}

	public Map<String, List<String>> getPropertiesIndexedByMap() {
		return propertiesIndexedByMap;
	}

	public List<ConfigurationItemDescriptorDto> getDeployables() {
		return copyOf(filter(descriptors, new Predicate<ConfigurationItemDescriptorDto>() {
			@Override
			public boolean apply(ConfigurationItemDescriptorDto input) {
				return input.getInterfaces().contains("udm.Deployable");
			}
		}));
	}

	public List<ConfigurationItemDescriptorDto> getDeployableArtifacts() {
		return copyOf(filter(getDeployables(), new Predicate<ConfigurationItemDescriptorDto>() {
			@Override
			public boolean apply(ConfigurationItemDescriptorDto input) {
				return input.getInterfaces().contains("udm.Artifact");
			}
		}));
	}

	public List<ConfigurationItemDescriptorDto> getDeployableResources() {
		return copyOf(filter(getDeployables(), new Predicate<ConfigurationItemDescriptorDto>() {
			@Override
			public boolean apply(ConfigurationItemDescriptorDto input) {
				return !input.getInterfaces().contains("udm.Artifact");
			}
		}));
	}


	public boolean isPropertyNameofType(final String type, final String propertyname) {
		final ConfigurationItemDescriptorDto configurationItemDescriptor = descriptorMap.get(type);
		if (configurationItemDescriptor == null) {
			return false;
		}
		return any(configurationItemDescriptor.getPropertyDescriptors(), new Predicate<ConfigurationItemPropertyDescriptorDto>() {
			@Override
			public boolean apply(@Nullable ConfigurationItemPropertyDescriptorDto input) {
				return input.getName().equals(propertyname);
			}
		});
	}

	public List<String> getPropertyName(String type) {
		Preconditions.checkNotNull(Strings.emptyToNull(type), "type should not be null or empty");
		Preconditions.checkArgument(descriptorMap.containsKey(type), "type %s not found", type);

		final ConfigurationItemDescriptorDto configurationItemDescriptorDto = descriptorMap.get(type);
		final List<ConfigurationItemPropertyDescriptorDto> propertyDescriptors = configurationItemDescriptorDto.getPropertyDescriptors();
		return copyOf(transform(propertyDescriptors, new Function<ConfigurationItemPropertyDescriptorDto, String>() {
			@Override
			public String apply(ConfigurationItemPropertyDescriptorDto input) {
				return input.getName();
			}
		}));
	}
}
