package com.xebialabs.deployit.client;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyDescriptorDto;

import javax.annotation.Nullable;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import static com.google.common.collect.ImmutableList.copyOf;
import static com.google.common.collect.Iterables.any;
import static com.google.common.collect.Iterables.filter;

public class Descriptors {

	private final List<ConfigurationItemDescriptorDto> descriptors;
	private final ImmutableMap<String, ConfigurationItemDescriptorDto> descriptorMap;

	public Descriptors(List<ConfigurationItemDescriptorDto> descriptors) {
		this.descriptors = Lists.newArrayList(descriptors);
		Collections.sort(this.descriptors, new Comparator<ConfigurationItemDescriptorDto>() {
			@Override
			public int compare(ConfigurationItemDescriptorDto o1, ConfigurationItemDescriptorDto o2) {
				return o1.getSimpleName().compareTo(o2.getSimpleName());
			}
		});

		descriptorMap = Maps.uniqueIndex(descriptors, new Function<ConfigurationItemDescriptorDto, String>() {
			@Override
			public String apply(ConfigurationItemDescriptorDto input) {
				return input.getSimpleName();
			}
		});
	}

	public List<ConfigurationItemDescriptorDto> getDescriptors() {
		return descriptors;
	}

	public List<ConfigurationItemDescriptorDto> getDeployables() {
		return copyOf(filter(descriptors, new Predicate<ConfigurationItemDescriptorDto>() {
			@Override
			public boolean apply(ConfigurationItemDescriptorDto input) {
				return input.getInterfaces().contains("udm.Deployable");
			}
		}));
	}

	public List<ConfigurationItemDescriptorDto> getDeployableArtifacts() {
		return copyOf(filter(getDeployables(), new Predicate<ConfigurationItemDescriptorDto>() {
			@Override
			public boolean apply(ConfigurationItemDescriptorDto input) {
				return input.getInterfaces().contains("udm.Artifact");
			}
		}));
	}

	public List<ConfigurationItemDescriptorDto> getDeployableResouces() {
		return copyOf(filter(getDeployables(), new Predicate<ConfigurationItemDescriptorDto>() {
			@Override
			public boolean apply(ConfigurationItemDescriptorDto input) {
				return !input.getInterfaces().contains("udm.Artifact");
			}
		}));
	}


	public boolean isPropertyNameofType(final String type, final String propertyname) {
		final ConfigurationItemDescriptorDto configurationItemDescriptor = descriptorMap.get(type);
		if (configurationItemDescriptor == null) {
			return false;
		}
		return any(configurationItemDescriptor.getPropertyDescriptors(), new Predicate<ConfigurationItemPropertyDescriptorDto>() {
			@Override
			public boolean apply(@Nullable ConfigurationItemPropertyDescriptorDto input) {
				return input.getName().equals(propertyname);
			}
		});
	}

}
