package com.xebialabs.deployit.integration.test;

import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import javax.script.ScriptException;

import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized.Parameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;

import com.xebialabs.deployit.cli.Cli;
import com.xebialabs.deployit.cli.CliOptions;
import com.xebialabs.deployit.cli.Interpreter;
import com.xebialabs.deployit.integration.test.support.TemporaryDirectoryHolder;

/**
 */
@RunWith(NiceParameterizedNames.class)
public class DeployitIntegrationTest {

    private static final String SETUP_FILE = "setUp.py";
    private static final String TEARDOWN_FILE = "tearDown.py";
    private static final String BASE_TEST_DIR = "src/test/jython";

    private static String testsToRunRegExpPattern = System.getProperty("DeployitIntegrationTest.testsToRunRegExpPattern", ".*\\.py");
    private static Pattern testsToRunPattern;

    static {
        testsToRunPattern = Pattern.compile(testsToRunRegExpPattern);
    }

    private static Cli cli;

    /* Small hack to skip all tests before the one named here. Null means no skipping, */
    private static String skipUntil = null;
//    private static String skipUntil = "exportArchivedTasksInDateRange.py";


    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();

    private Interpreter interpreter;
    private File scriptFile;

    public DeployitIntegrationTest(File scriptFileToRun) {
        this.scriptFile = scriptFileToRun;
    }

    /**
     * Returns a List<Object[]> which contains:
     * - the file to run
     * - whether to run or ignore the test.
     */
    @Parameters
    public static List<Object[]> scriptsToRun() {
        final String baseDir = System.getProperty("jythonTestBaseDir", BASE_TEST_DIR);
        logger.info("Running all tests in {}", baseDir);
        List<Object[]> targets = new ArrayList<Object[]>();
        scriptDir(new File(baseDir), targets);
        return targets;
    }

    private static void scriptDir(File testsDir, final List<Object[]> targets) {
        if (testsDir.exists()) {
            File[] scriptFiles = testsDir.listFiles(new FilenameFilter() {
                @Override
                public boolean accept(File dir, String name) {
                    final boolean isDirectory = new File(dir, name).isDirectory();
                    boolean testFrameworkFile = SETUP_FILE.equals(name) || TEARDOWN_FILE.equals(name);
                    boolean correctExtension = checkExtension(name, ".py") || checkExtension(name, ".cli");
                    return isDirectory || (correctExtension && !testFrameworkFile);
                }
            });
            for (File eachFile : scriptFiles) {
                if (eachFile.isDirectory()) {
                    scriptDir(eachFile, targets);
                } else {
                    addTarget(targets, eachFile);
                }
            }
        }
    }

    private static boolean checkExtension(String name, String suffix) {
        return name.endsWith(suffix) || name.endsWith(suffix + "ignore");
    }

    private static void addTarget(final List<Object[]> targets, final File eachFile) {
        if (testsToRunPattern.matcher(eachFile.getName()).matches()) {
            logger.info("Adding testcase for {}", eachFile);
            targets.add(new Object[]{eachFile, !eachFile.getName().endsWith("ignore")});
        } else {
            logger.info("Ignoring testcase for {}", eachFile);
        }
    }

    @BeforeClass
    public static void init() throws Exception {
        CliOptions options = new CliOptions();
        options.setHost("localhost");
        options.setPort(9696);
        options.setExposeProxies(true);
        options.setUsername("admin");
        options.setPassword("admin");

        cli = new Cli(options);
    }

    private static void cleanupTaskBackup() {
        File recoveryDir = new File("work");
        if (recoveryDir.exists()) {
            for (File file : recoveryDir.listFiles()) {
                if (file.getName().endsWith(".task")) {
                    file.delete();
                }
            }
        }
    }

    @Test
    public void testCliScripts() throws Exception {
        // Fast-forward to desired script
        if (skipCurrentTest()) {
            return;
        }
        skipUntil = null;

        // Execute script
        TemporaryDirectoryHolder.init(temporaryFolder.getRoot());
        try {
            interpreter.evaluate(new FileReader(scriptFile));
        } catch (Exception e) {
            logger.error("Exception while executing script " + scriptFile, e);
            throw new Exception("Errors found during executing " + scriptFile.getName(), e);
        } finally {
            TemporaryDirectoryHolder.destroy();
        }
    }

    private boolean skipCurrentTest() {
        return skipUntil != null && !skipUntil.equals(scriptFile.getName());
    }

    @Before
    public void setupInterpreterAndCheckCleanRepository() throws IOException, ScriptException {
        if (skipCurrentTest()) {
            logger.info("Skipping {}", scriptFile.getName());
            return;
        }
        logger.info("Executing {}", scriptFile.getName());

        interpreter = cli.getNewInterpreter();
        checkClean();
        cleanupTaskBackup();
        scanForFileInParentDirectoriesAndExecute(SETUP_FILE, scriptFile.getParentFile());
    }

    private void scanForFileInParentDirectoriesAndExecute(String fileNameToScanUpFor, File parentDir) throws ScriptException, FileNotFoundException {
        File setUp = new File(parentDir, fileNameToScanUpFor);
        logger.info(parentDir + ": " + setUp);
        final String lastPartOfBaseTestDir = BASE_TEST_DIR.substring(BASE_TEST_DIR.lastIndexOf('/') + 1);
        while (!setUp.exists()) {
            if (parentDir.getAbsolutePath().endsWith(lastPartOfBaseTestDir)) {
                break;
            }
            parentDir = parentDir.getParentFile();
            setUp = new File(parentDir, SETUP_FILE);
            logger.info(parentDir + ": " + setUp);
        }

        if (setUp.exists()) {
            interpreter.evaluate(new FileReader(setUp));
        }
    }

    private void checkClean() throws ScriptException, IOException {
        interpreter.evaluate(new InputStreamReader(new ClassPathResource("checkClean.py").getInputStream()));
    }

    @After
    public void tearDown() throws ScriptException, IOException {
        if (skipCurrentTest()) {
            return;
        }

        scanForFileInParentDirectoriesAndExecute(TEARDOWN_FILE, scriptFile.getParentFile());
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployitIntegrationTest.class);
}
