package com.xebialabs.deployit.integration.test;

import static com.xebialabs.deployit.jcr.JcrConstants.ADMIN_PASSWORD;
import static com.xebialabs.deployit.jcr.JcrConstants.ADMIN_USERNAME;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

import javax.script.ScriptException;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized.Parameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.FileSystemResource;

import com.google.common.io.Files;
import com.xebialabs.deployit.DeployitConfig;
import com.xebialabs.deployit.DeployitOptions;
import com.xebialabs.deployit.Server;
import com.xebialabs.deployit.cli.Cli;
import com.xebialabs.deployit.cli.CliOptions;
import com.xebialabs.deployit.cli.Interpreter;
import com.xebialabs.deployit.integration.test.support.TemporaryDirectoryHolder;
import com.xebialabs.deployit.jcr.JackrabbitRepositoryFactoryBean;

/**
 */
@RunWith(NiceParameterizedNames.class)
public class DeployitIntegrationTest {

	private static final String SETUP_FILE = "setUp.py";
	private static final String TEARDOWN_FILE = "tearDown.py";
	private static final String BASE_TEST_DIR = "src/test/jython";

	@Rule
	public TemporaryFolder temporaryFolder = new TemporaryFolder();

	private static File repositoryHomeDir;

	Interpreter interpreter;

	private File scriptFile;

    private static String filesToRunPattern = System.getProperty("DeployitIntegrationTest.filesToRunRegExpPattern",".*\\.py");;

    private static Cli cli;

	public DeployitIntegrationTest(File scriptFileToRun) {
		this.scriptFile = scriptFileToRun;
	}

    /**
     * Returns a List<Object[]> which contains:
     * - the file to run
     * - whether to run or ignore the test.
     */
	@Parameters
	public static List<Object[]> scriptsToRun() {
		final String baseDir = System.getProperty("jythonTestBaseDir", BASE_TEST_DIR);
		logger.info("Running all tests in {}", baseDir);
		List<Object[]> targets = new ArrayList<Object[]>();
		scriptDir(new File(baseDir), targets);
		return targets;
	}

	private static void scriptDir(File testsDir, final List<Object[]> targets) {
		if (testsDir.exists()) {
			File[] scriptFiles = testsDir.listFiles(new FilenameFilter() {
				@Override
				public boolean accept(File dir, String name) {
					final boolean isDirectory = new File(dir, name).isDirectory();
                    boolean testFrameworkFile = SETUP_FILE.equals(name) || TEARDOWN_FILE.equals(name);
                    boolean correctExtension = checkExtension(name, ".py") || checkExtension(name, ".cli");
                    return isDirectory || (correctExtension && !testFrameworkFile);
				}
			});
			for (File eachFile : scriptFiles) {
				if (eachFile.isDirectory()) {
					scriptDir(eachFile, targets);
				} else {
					addTarget(targets, eachFile);
				}
			}
		}
	}

	private static boolean checkExtension(String name, String suffix) {
        return name.endsWith(suffix) || name.endsWith(suffix + "ignore");
    }

    private static void addTarget(final List<Object[]> targets, final File eachFile) {
        logger.info("Adding testcase for {}", eachFile);
        targets.add(new Object[] { eachFile, !eachFile.getName().endsWith("ignore") });
    }

    @BeforeClass
	public static void init() throws Exception {
		setupJcrRepository();
        cleanupTaskBackup();

		DeployitOptions deployitOptions = new DeployitOptions();
		deployitOptions.setTestModeEnabled(true);
        Server.start(deployitOptions);

	    DeployitConfig loadForUse = DeployitConfig.loadForUse();
	    CliOptions options = new CliOptions();
	    options.setHost("localhost");
	    options.setPort(loadForUse.getHttpPort());
	    options.setExposeProxies(true);
	    options.setUsername(ADMIN_USERNAME);
	    options.setPassword(ADMIN_PASSWORD);

	    cli = new Cli(options);
	}

    private static void cleanupTaskBackup() {
        File recoveryFile = new File("recovery.dat");
        if (recoveryFile.exists()) {
        	assertThat(recoveryFile.delete(), is(true));
        }
    }

    private static void setupJcrRepository() throws Exception {
		repositoryHomeDir = new File("repository");
	    if (repositoryHomeDir.exists()) {
			Files.deleteRecursively(repositoryHomeDir);
	    }
		String homeDirAbsolutePath = repositoryHomeDir.getAbsolutePath();

		JackrabbitRepositoryFactoryBean repositoryFactoryBean = new JackrabbitRepositoryFactoryBean();
		repositoryFactoryBean.setConfiguration(new ClassPathResource("jackrabbit-repository.xml"));
		repositoryFactoryBean.setHomeDir(new FileSystemResource(homeDirAbsolutePath));

		repositoryFactoryBean.setCreateHomeDirIfNotExists(true);
		repositoryFactoryBean.afterPropertiesSet();
		repositoryFactoryBean.configureJcrRepositoryForDeployit();
		repositoryFactoryBean.destroy();
	}

	@AfterClass
	public static void shutdown() {
		Server.requestShutdown();
	}

	@Test
	public void testCliScripts() throws Exception {
		TemporaryDirectoryHolder.init(temporaryFolder.getRoot());
		logger.info("Executing script file " + scriptFile.getName());
		try {
			interpreter.evaluate(new FileReader(scriptFile));
		} catch (Exception e) {
			logger.error("Exception while executing script " + scriptFile, e);
			throw new Exception("Errors found during executing " + scriptFile.getName(), e);
		} finally {
			TemporaryDirectoryHolder.destroy();
		}
	}

	@Before
	public void setupInterpreterAndCheckCleanRepository() throws IOException, ScriptException {
		interpreter = cli.getNewInterpreter();
		checkClean();
		scanForFileInParentDirectoriesAndExecute(SETUP_FILE, scriptFile.getParentFile());
	}

	private void scanForFileInParentDirectoriesAndExecute(String fileNameToScanUpFor, File parentDir) throws ScriptException, FileNotFoundException {
		File setUp = new File(parentDir, fileNameToScanUpFor);
		logger.info(parentDir + ": " + setUp);
		final String lastPartOfBaseTestDir = BASE_TEST_DIR.substring(BASE_TEST_DIR.lastIndexOf('/') + 1);
		while (!setUp.exists()) {
			logger.info(parentDir.getAbsolutePath());
			if (parentDir.getAbsolutePath().endsWith(lastPartOfBaseTestDir)) {
				break;
			}
			parentDir = parentDir.getParentFile();
			setUp = new File(parentDir, SETUP_FILE);
			logger.info(parentDir + ": " + setUp);
		}

		if (setUp.exists()) {
			interpreter.evaluate(new FileReader(setUp));
		}
	}

	private void checkClean() throws ScriptException, IOException {
		interpreter.evaluate(new InputStreamReader(new ClassPathResource("checkClean.py").getInputStream()));
	}

	@After
	public void tearDown() throws ScriptException, IOException {
		scanForFileInParentDirectoriesAndExecute(TEARDOWN_FILE, scriptFile.getParentFile());
	}

	private static final Logger logger = LoggerFactory.getLogger(DeployitIntegrationTest.class);
}
