package com.xebialabs.deployit.cli;

import com.xebialabs.deployit.booter.remote.BooterConfig;
import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.RemoteBooter;
import com.xebialabs.deployit.cli.api.ProxiesInstance;
import com.xebialabs.deployit.cli.help.HelpScanner;
import com.xebialabs.deployit.cli.util.Closeables;
import com.xebialabs.deployit.cli.util.PasswordEncrypter;
import jline.Terminal;
import jline.TerminalFactory;
import jline.console.ConsoleReader;
import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.script.ScriptException;
import java.io.*;
import java.lang.reflect.Constructor;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import static com.xebialabs.deployit.cli.CliOptions.EXTENSION_DIR;
import static com.xebialabs.deployit.cli.CliOptions.parse;
import static java.util.stream.Collectors.toSet;

public class Cli {

    public static final String CLI_PASSWORD_PROPERTY_NAME = "cli.password";
    public static final String CLI_USERNAME_PROPERTY_NAME = "cli.username";

    private static final AtomicReference<Properties> properties = new AtomicReference<>();
    private ConsoleReader consoleReader;
    private static final FilenameFilter CLI_EXTENSION_FILTER = new FilenameFilter() {
        @Override
        public boolean accept(final File dir, final String name) {
            return name.endsWith(".py") || name.endsWith(".cli");
        }
    };
    private ScriptEngineBuilder scriptEngine;
    private CliOptions options;
    private DeployitCommunicator communicator;
    private PasswordEncrypter encrypter;
    private Set<String> cliObjectNames;

    public Cli(CliOptions options) throws Exception {
        this(options, null);
    }

    public Cli(CliOptions options, ConsoleReader consoleReader) throws Exception {
        this.options = options;
        this.consoleReader = consoleReader;
        scriptEngine = new ScriptEngineBuilder();
        initialize();
    }

    private void printBanner() {
        bannerPrint("Welcome to the XL Deploy Jython CLI!");
        bannerPrint("Type 'help' to learn about the objects you can use to interact with XL Deploy.");
        bannerPrint("");
    }

    private void initialize() throws Exception {
        PasswordEncrypter.init();
        encrypter = PasswordEncrypter.getInstance();

        BooterConfig.Builder builder = BooterConfig.builder().withHost(options.getHost()).withContext(options.getContext()).withPort(options.getPort())
                .withProxyHost(options.getProxyHost()).withProxyPort(options.getProxyPort()).withSocketTimeout(options.getSocketTimeout());
        if (options.isSecured()) {
            builder.withProtocol(BooterConfig.Protocol.HTTPS);
        }
        createCredentials(builder);
        if (consoleReader != null) {
            consoleReader.addCompleter(new CliCompleter(consoleReader));
        }
        BooterConfig config = builder.build();
        RemoteBooter.boot(config);
        communicator = RemoteBooter.getCommunicator(config);
        printBanner();
        ProxiesInstance proxies = createAndRegisterProxies(config);
        registerCliObjects(proxies);
    }

    private ProxiesInstance createAndRegisterProxies(BooterConfig config) {
        ProxiesInstance proxies = new ProxiesInstance(config);
        if (options.isExposeProxies()) {
            System.out.println("Exposing Proxies!");
            scriptEngine.put("proxies", proxies);
        }
        return proxies;
    }

    public static void main(String[] args) throws Exception {
        CliOptions options = parse(args);
        if (options == null) {
            return;
        }
        new Cli(options, setupConsole()).getNewInterpreter().interpret();
    }

    private static ConsoleReader setupConsole() throws IOException {
        final Terminal terminal = getTerminal();
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                try {
                    terminal.restore();
                } catch (Exception ignored) {
                }
            }
        });
        ConsoleReader cr = new ConsoleReader();
        cr.setExpandEvents(false);
        return cr;
    }

    private static Terminal getTerminal() {
        Terminal terminal;
        try {
            terminal = TerminalFactory.create();
        } catch (Throwable t) {
            System.err.println("[WARNING] Error loading terminal, using fallback. Your terminal will have reduced functionality. Please see: http://docs.xebialabs.com/general/faq.html#fallback-terminal");
            logger.warn("Error loading terminal, using fallback. Your terminal will have reduced functionality.");
            TerminalFactory.configure(TerminalFactory.Type.NONE);
            terminal = TerminalFactory.create();
        }
        return terminal;
    }

    public Interpreter getNewInterpreter() throws ScriptException, FileNotFoundException {
        final Interpreter interpreter = new Interpreter(consoleReader, scriptEngine, communicator, options, cliObjectNames);
        readExtensions(interpreter);
        return interpreter;
    }

    private void registerCliObjects(final ProxiesInstance proxies) throws Exception {
        final Scannit scannit = new Scannit(Configuration.config().with(new TypeAnnotationScanner()).scan("com.xebialabs"));
        final Set<Class<?>> classes = scannit.getTypesAnnotatedWith(CliObject.class);
        cliObjectNames = classes.stream().map(cliObject -> cliObject.getAnnotation(CliObject.class).name()).collect(toSet());
        for (Class<?> cliObject : classes) {
            final Constructor<?> constructor = cliObject.getConstructor(ProxiesInstance.class);
            final Object o = constructor.newInstance(proxies);
            final String name = cliObject.getAnnotation(CliObject.class).name();
            scriptEngine.put(name, o);
        }
        HelpScanner.init(classes);
        if (!options.isQuiet()) {
            HelpScanner.printHelp();
        }
    }

    public void createCredentials(BooterConfig.Builder builder) throws IOException {
        builder.withCredentials(retrieveUsername(), retrievePassword());
    }

    private String retrieveUsername() throws IOException {
        String username;
        if (options.isUsernameOnCommandline()) {
            username = options.getUsername();
        } else if (deployitConfigurationFileExists()) {
            username = readFromProperties(CLI_USERNAME_PROPERTY_NAME);
        } else {
            username = consoleReader.readLine("Username: ");
        }
        return username;
    }

    private String retrievePassword() throws IOException {
        String password;
        if (options.isPasswordOnCommandline()) {
            password = options.getPassword();
        } else if (deployitConfigurationFileExists()) {
            password = encrypter.ensureDecrypted(readFromProperties(CLI_PASSWORD_PROPERTY_NAME));
        } else {
            password = consoleReader.readLine("Password: ", '\0');
        }
        return password;
    }

    private boolean deployitConfigurationFileExists() {
        return options.getConfigurationFile().exists();
    }

    private String readFromProperties(final String key) throws IOException {
        if (properties.get() == null) {
            Properties props = readDeployitConfigurationFile();
            properties.set(props);
        }

        return properties.get().getProperty(key);
    }

    private Properties readDeployitConfigurationFile() throws IOException {
        return readDeployitConfigurationFile(options, encrypter);
    }

    static Properties readDeployitConfigurationFile(CliOptions options, PasswordEncrypter encrypter) throws IOException {
        Properties props = new Properties();

        try (FileInputStream inStream = new FileInputStream(options.getConfigurationFile())) {
            props.load(inStream);
        }

        String cliPassword = props.getProperty(CLI_PASSWORD_PROPERTY_NAME);
        if (cliPassword != null && !encrypter.isEncrypted(cliPassword)) {
            cliPassword = encrypter.encrypt(cliPassword);
            props.setProperty(CLI_PASSWORD_PROPERTY_NAME, cliPassword);

            if (options.getConfigurationFile().canWrite()) {
                logger.info("Writing configuration file [{}] with encrypted " + CLI_PASSWORD_PROPERTY_NAME, options.getConfigurationFile());
                FileOutputStream outStream = new FileOutputStream(options.getConfigurationFile());
                try {
                    props.store(outStream, "XL Deploy CLI configuration file");
                } catch (IOException exc) {
                    logger.error("Error while updating configuration file [{}]: [{}]", options.getConfigurationFile(), exc.getMessage());
                } finally {
                    Closeables.closeQuietly(outStream);
                }
                logger.info("Done writing configuration file [{}] with encrypted " + CLI_PASSWORD_PROPERTY_NAME, options.getConfigurationFile());
            } else {
                logger.warn("Cannot write configuration file [{}], not encrypting " + CLI_PASSWORD_PROPERTY_NAME, options.getConfigurationFile());
            }
        }

        return props;
    }

    private void readExtensions(Interpreter interpreter) throws ScriptException, FileNotFoundException {
        final File extensionDir = new File(EXTENSION_DIR);
        if (!extensionDir.exists() || !extensionDir.isDirectory()) {
            System.out.println("No extension directory present.");
            return;
        }

        final File[] files = extensionDir.listFiles(CLI_EXTENSION_FILTER);

        for (File extension : files) {
            bannerPrint("Reading extension: " + extension);
            interpreter.evaluate(new FileReader(extension));
        }
    }

    void bannerPrint(String line) {
        if (!options.isQuiet()) {
            System.out.println(line);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(Cli.class);
}
