package com.xebialabs.deployit.cli.api;

import java.io.InputStream;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.engine.api.dto.ArtifactAndData;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;

import static com.google.common.base.Preconditions.checkArgument;

@CliObject(name = "factory")
@ClassHelp(description = "Helper that can construct Configuration Items (CI) and Artifacts")
public class ObjectFactory extends DocumentedObject {

    public ObjectFactory(ProxiesInstance proxiesInstance) {
    }

    public ObjectFactory() {
    }

    @MethodHelp(description = "Construct a CI of a specified type", parameters = {
        @ParameterHelp(name = "id", description = "The id of the CI"),
        @ParameterHelp(name = "ciType", description = "The type of the CI")
    })
    public ConfigurationItem configurationItem(String id, String ciType) {
        return ci(id, ciType, Maps.<String, Object> newHashMap());
    }

    @MethodHelp(description = "Construct a CI of a specified type with specified values", parameters = {
        @ParameterHelp(name = "id", description = "The id of the CI"),
        @ParameterHelp(name = "ciType", description = "The type of the CI"),
        @ParameterHelp(name = "values", description = "The values of the CI in a map.")
    })
    public ConfigurationItem configurationItem(final String id, final String ciType, final Map<String, Object> values) {
        return ci(id, ciType, values);
    }

    private static ConfigurationItem ci(final String id, final String ciType, final Map<String, Object> values) {
        checkArgument(isValidType(ciType), "Unknown CI type %s", ciType);

        final Type type = Type.valueOf(ciType);
        ConfigurationItem configurationItem = type.getDescriptor().newInstance(id);
        // Need to wrap the incoming map into a hashmap, as it could be a PyDict.
        for (String s : values.keySet()) {
            PropertyDescriptor propertyDescriptor = type.getDescriptor().getPropertyDescriptor(s);
            if (propertyDescriptor != null) {
                propertyDescriptor.set(configurationItem, values.get(s));
            }
        }
        return configurationItem;
    }

    @MethodHelp(description = "Construct an artifact of a specific type with associated data", parameters = {
            @ParameterHelp(name = "id", description = "The id of the artifact"),
            @ParameterHelp(name = "ciType", description = "The type of the artifact"),
            @ParameterHelp(name = "values", description = "The values for the artifact in a map."),
            @ParameterHelp(name = "data", description = "The data of the artifact that is to be uploaded")
    }, returns = "An Artifact with data that can be stored in XL Deploy.",
            deprecated = "Please use artifactAsInputStream(String id, String ciType, Map<String, Object> values, InputStream data)")
    public ArtifactAndData artifact(String id, String ciType, Map<String, Object> values, byte[] data) {
        ConfigurationItem ci = ci(id, ciType, values);
        checkArgument(ci instanceof Artifact, "The type should be an Artifact (was: [%s], type: [%s])", ci.getClass(), ciType);
        return new ArtifactAndData((Artifact) ci, data);
    }

    @MethodHelp(description = "Construct an artifact of a specific type with associated data", parameters = {
        @ParameterHelp(name = "id", description = "The id of the artifact"),
        @ParameterHelp(name = "ciType", description = "The type of the artifact"),
        @ParameterHelp(name = "values", description = "The values for the artifact in a map."),
        @ParameterHelp(name = "data", description = "The data of the artifact that is to be uploaded")
    }, returns = "An Artifact with data that can be stored in XL Deploy.")
    public ArtifactAndData artifactAsInputStream(String id, String ciType, Map<String, Object> values, InputStream data) {
        ConfigurationItem ci = ci(id, ciType, values);
        checkArgument(ci instanceof Artifact, "The type should be an Artifact (was: [%s], type: [%s])", ci.getClass(), ciType);
        return new ArtifactAndData((Artifact) ci, data);
    }

    @MethodHelp(description = "Prints all registered configuration item types.")
    public void types() {
        final Set<String> typeSet = com.google.common.collect.Sets.newTreeSet(Collections2.transform(DescriptorRegistry.getDescriptors(),
            new Function<Descriptor, String>() {
                public String apply(Descriptor input) {
                    return input.getType().toString();
                }
            }));
        System.out.printf("The registered configuration item types are:\n%s\n\n", typeSet);
    }

    public static Type toType(String ciType) {
        if (ciType == null) {
            return null;
        }

        return Type.valueOf(ciType);
    }

    private static boolean isValidType(final String ciType) {
        return DescriptorRegistry.exists(Type.valueOf(ciType));
    }

}
