package com.xebialabs.deployit.cli.help;

import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;

import com.xebialabs.deployit.cli.CliObject;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newTreeSet;
import static java.lang.System.out;

public class HelpScanner {

    private static final AtomicReference<Set<Class<?>>> discoveredCliObjects = new AtomicReference<>();

    public static void init(Set<Class<?>> clazzes) {
        discoveredCliObjects.set(clazzes);
    }

    public static List<String> loadCliObjectCompletions() {
        List<String> completions = new ArrayList<>();
        Set<Class<?>> availableCliObjects = loadAvailableCliObjects();
        for (Class<?> clazz : availableCliObjects) {
            final CliObject cliObject = clazz.getAnnotation(CliObject.class);
            completions.add(cliObject.name());
        }
        return completions;
    }

    private static Set<Class<?>> loadAvailableCliObjects() {
        Set<Class<?>> availableCliObjects = newTreeSet(new Comparator<Class<?>>() {
            public int compare(Class<?> clazz, Class<?> clazzComparedTo) {
                return clazz.getSimpleName().compareTo(clazzComparedTo.getSimpleName());
            }
        });
        if (discoveredCliObjects.get() != null) {
            availableCliObjects.addAll(discoveredCliObjects.get());
        }
        return availableCliObjects;
    }

    public static List<String> loadObjectMethodCompletions(String objectName) {
        List<String> completions = new ArrayList<>();
        Set<Class<?>> availableCliObjects = loadAvailableCliObjects();
        for (Class<?> cliClazz : availableCliObjects) {
            final CliObject cliObject = cliClazz.getAnnotation(CliObject.class);
            if (objectName.equals(cliObject.name())) {
                List<String> completionsForObject = loadObjectMethodCompletionsForObject(cliObject, cliClazz);
                completions.addAll(completionsForObject);
            }
        }
        return completions;
    }

    private static List<String> loadObjectMethodCompletionsForObject(CliObject cliObject, Class<?> cliClazz) {
        List<String> completions = new ArrayList<>();
        String objectName = cliObject.name();
        List<Method> availableMethods = newArrayList(cliClazz.getDeclaredMethods());
        for (Method method : availableMethods) {
            String help = loadMethodSignatureHelp(objectName, method);
            if (help != null) {
                String suggestion = cliObject.name() + "." + method.getName();
                if (!completions.contains(suggestion)) {
                    completions.add(suggestion);
                }
            }
        }
        return completions;
    }

    public static List<String> loadMethodSignatureSuggestions(String objectName, String methodName) {
        List<String> helpSuggestions = new ArrayList<>();
        Set<Class<?>> availableCliObjects = loadAvailableCliObjects();
        for (Class<?> cliClazz : availableCliObjects) {
            final CliObject cliObject = cliClazz.getAnnotation(CliObject.class);
            if (objectName.equals(cliObject.name())) {
                List<String> suggestionsForObject = loadMethodSignatureSuggestionsForObject(objectName, cliClazz, methodName);
                helpSuggestions.addAll(suggestionsForObject);
                break;
            }
        }
        return helpSuggestions;
    }

    private static List<String> loadMethodSignatureSuggestionsForObject(String objectName, Class<?> cliClazz, String methodName) {
        List<String> helpSuggestions = new ArrayList<>();
        List<Method> availableMethods = newArrayList(cliClazz.getDeclaredMethods());
        for (Method method : availableMethods) {
            if (methodName.equals(method.getName())) {
                String help = loadMethodSignatureHelp(objectName, method);
                if (help != null) {
                    helpSuggestions.add(help);
                }
            }
        }
        return helpSuggestions;
    }

    public static void printHelp() {
        Set<Class<?>> availableCliObjects = loadAvailableCliObjects();
        out.println("XL Deploy Objects available on the CLI:\n");
        for (Class<?> clazz : availableCliObjects) {
            final ClassHelp classHelp = clazz.getAnnotation(ClassHelp.class);
            final CliObject cliObject = clazz.getAnnotation(CliObject.class);
            if (classHelp != null) {
                out.printf("* %s: %s\n", cliObject.name(), classHelp.description());
            }
        }
        out.println("\nTo know more about a specific object, type <objectname>.help()");
        out.println("To get to know more about a specific method of an object, type <objectname>.help(\"<methodname>\")\n");
    }

    public static void printHelp(Class<?> clazz) {
        final ClassHelp classHelp = clazz.getAnnotation(ClassHelp.class);
        final CliObject cliObject = clazz.getAnnotation(CliObject.class);
        if (classHelp != null) {
            final String objectName = cliObject.name();
            out.printf("%s: %s\n\n", objectName, classHelp.description());
            printMethods(objectName, clazz);
            out.println();
        } else {
            out.println("Not found help for " + clazz);
        }
    }

    public static void printHelp(Class<?> clazz, String methodName) {
        final ClassHelp classHelp = clazz.getAnnotation(ClassHelp.class);
        final CliObject cliObject = clazz.getAnnotation(CliObject.class);
        if (classHelp != null) {
            final String objectName = cliObject.name();
            for (Method method : clazz.getMethods()) {
                if (method.getName().equals(methodName)) {
                    printMethod(objectName, method);
                    printMethodDetails(method);
                    out.println();
                }
            }
        }
    }

    private static void printMethodDetails(final Method method) {
        final MethodHelp methodHelp = method.getAnnotation(MethodHelp.class);
        if (methodHelp == null) {
            return;
        }

        out.println();
        if (!isNullOrEmpty(methodHelp.deprecated())) {
            out.printf("DEPRECATED - %s\n\n", methodHelp.deprecated());
        }
        out.printf("Description:\n%s\n\n", methodHelp.description());
        out.print("Parameters:\n");
        if (methodHelp.parameters().length != 0) {
            for (ParameterHelp parameterHelp : methodHelp.parameters()) {
                out.printf("  %s: %s\n", parameterHelp.name(), parameterHelp.description());
            }
        } else {
            out.print("None\n");
        }
        out.println();
        out.printf("Returns:\n%s - %s\n", method.getReturnType().getSimpleName(), methodHelp.returns());
    }

    private static void printMethods(final String objectName, final Class<?> clazz) {
        List<Method> availableMethods = newArrayList(clazz.getDeclaredMethods());
        Collections.sort(availableMethods, new Comparator<Method>() {
            public int compare(Method method, Method methodComparedTo) {
                if (method.getName().equals(methodComparedTo.getName())) {
                    return 0;
                    // FIXME : if 0 then compare on number of parameters, else on length of parameters?
                } else {
                    return method.getName().compareTo(methodComparedTo.getName());
                }
            }
        });
        out.println("The methods available are:");
        for (Method method : availableMethods) {
            printMethod(objectName, method);
        }
    }

    private static void printMethod(final String objectName, final Method method) {
        String help = loadMethodSignatureHelp(objectName, method);
        if (help != null) {
            out.printf("* %s", help);
        }
    }

    private static String loadMethodSignatureHelp(final String objectName, final Method method) {
        final MethodHelp methodHelp = method.getAnnotation(MethodHelp.class);
        if (methodHelp == null) {
            return null;
        }

        String methodName = method.getName();
        StringBuilder params = new StringBuilder();
        final Class<?>[] classes = method.getParameterTypes();
        final ParameterHelp[] parameterHelps = methodHelp.parameters();
        if (classes.length != parameterHelps.length) {
            throw new IllegalArgumentException("Not all parameters are documented!");
        }

        for (int i = 0; i < classes.length; i++) {
            params.append(classes[i].getSimpleName());
            params.append(" ").append(parameterHelps[i].name());
            if (i < classes.length - 1) {
                params.append(", ");
            }
        }

        String deprecationWarning = "";
        if (!isNullOrEmpty(methodHelp.deprecated())) {
            deprecationWarning = "- DEPRECATED";
        }

        return String.format("%s.%s(%s) : %s %s\n", objectName, methodName, params, method.getReturnType().getSimpleName(), deprecationWarning);
    }

}
