package com.xebialabs.deployit.cli;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.Reader;

import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptException;
import javax.script.SimpleScriptContext;

import jline.console.ConsoleReader;

import org.python.core.Py;
import org.python.core.PyBaseExceptionDerived;
import org.python.core.PyException;

import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.cli.help.HelpScanner;
import com.xebialabs.deployit.cli.util.Closeables;

import static java.lang.System.out;

public class Interpreter {
    private ScriptEngine scriptEngine;
    private CliOptions options;
    private ScriptContext scriptContext;
    private ConsoleReader consoleReader;
    private DeployitCommunicator communicator;

    public Interpreter(ConsoleReader consoleReader, ScriptEngineBuilder scriptEngineBuilder, DeployitCommunicator communicator, CliOptions options) {
        this.consoleReader = consoleReader;
        this.communicator = communicator;
        this.scriptEngine = scriptEngineBuilder.build(options);
        this.options = options;
        this.scriptContext = new SimpleScriptContext();
        ScriptContext context = scriptEngine.getContext();
        this.scriptContext.setBindings(context.getBindings(ScriptContext.ENGINE_SCOPE), ScriptContext.GLOBAL_SCOPE);
    }

    void interpret() throws Exception {
        try {
            if (options.interpretSourceFile()) {
                scriptContext.setAttribute(ScriptEngine.FILENAME, options.getSourceFileName(), ScriptContext.ENGINE_SCOPE);
                FileReader reader = new FileReader(options.getSourceFileName());
                try {
                    evaluate(reader);
                } finally {
                    Closeables.closeQuietly(reader);
                }
            } else {
                try {
                    advancedRepl(consoleReader);
                } catch (Exception e) {
                    throw new IllegalStateException("The REPL gave an exception!", e);
                }
            }
        } catch (FileNotFoundException e) {
            System.err.println("File " + options.getSourceFileName() + " could not be read, reason:" + e.getMessage());
            System.exit(1);
        } catch (ScriptException e) {
            int exitCode;
            if (sysExitInvoked(e)) {
                exitCode = getSysExitCode(e);
            } else {
                printScriptException(e);
                exitCode = 1;
            }
            System.exit(exitCode);
        }
    }

    private boolean sysExitInvoked(ScriptException e) {
        return e.getCause() instanceof PyException && ((PyException) e.getCause()).type.equals(Py.SystemExit);
    }

    private int getSysExitCode(ScriptException e) {
        return ((PyBaseExceptionDerived) ((PyException) e.getCause()).value).message.asInt();
    }

    public Object evaluate(Reader reader) throws ScriptException {
        return scriptEngine.eval(reader, scriptContext);
    }

    Object evaluate(String expression) throws ScriptException {
        if (expression == null) {
            return null;
        }
        return scriptEngine.eval(expression, scriptContext);
    }

    private void advancedRepl(ConsoleReader consoleReader) throws Exception {
        for (;;) {
            final String line = consoleReader.readLine(getPrompt());
            final String trimmed = line != null ? line.trim() : null;
            if (trimmed == null || trimmed.startsWith("quit") || trimmed.startsWith("exit")) {
                exit();
                break;
            } else if (trimmed.startsWith("help")) {
                HelpScanner.printHelp();
                continue;
            }
            try {
                Object evaluationResult = evaluate(line);
                if (evaluationResult instanceof String) {
                    out.println((String)evaluationResult);
                } else if (evaluationResult != null) {
                    out.println(evaluationResult.getClass().getSimpleName() + ": " + evaluationResult);
                } else {
                    //FIXME: not sure what to give back; maybe out.println("Ok"); ?
                }
            } catch (ScriptException e) {
                printScriptException(e);
            }
        }
    }

    private void printScriptException(ScriptException e) {
        Throwable cause = e.getCause();
        if (cause == null) {
            System.err.println(e);
        } else if (cause instanceof PyException) {
            if (cause.getCause() == null) {
                System.err.println(cause);
            } else {
                System.err.println(cause.getCause());
            }
        } else {
            System.err.println(cause);
        }
    }

    protected void exit() throws Exception {
        System.out.println("Bye bye!");
    }

    String getPrompt() {
        String prompt = "deployit > ";
        String username = communicator.getHttpClientHolder().getUserName();
        if (username != null) {
            prompt = username + " > ";
        }
        return prompt;
    }

}
