package com.xebialabs.deployit.cli.api;

import java.util.List;

import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.engine.api.TaskService;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.engine.api.execution.TaskWithSteps;

import static com.xebialabs.deployit.cli.help.DateHelp.toLocalDate;

/**
 * Exposes the {@link TaskService}.
 *
 * @since Deployit 3.8
 */
@CliObject(name = "tasks")
@ClassHelp(description = "Access to the task engine of Deployit.")
public class TaskClient extends DocumentedObject {

    private final TaskService service;

    public TaskClient() {
        service = null;
    }

    public TaskClient(ProxiesInstance proxies) {
        service = proxies.getTaskRegistry();
    }

    //
    // Method delegates to TaskService
    //

    @MethodHelp(description =
        "Lists the active tasks of the logged in user.",
        returns = "A list of tasks without step information")
    public List<TaskState> getMyCurrentTasks() {
        return service.getMyCurrentTasks();
    }

    @MethodHelp(description =
        "Lists the active tasks of all users. Requires admin permission.",
        returns = "A list of tasks without step information")
    public List<TaskState> getAllCurrentTasks() {
        return service.getAllCurrentTasks();
    }

    @MethodHelp(description =
        "Gets the task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") },
        returns = "The task without step information.")
    public TaskState get(String taskId) {
        return service.getTask(taskId);
    }

    @MethodHelp(description =
        "Gets the task with steps.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") },
        returns = "The task, containing step information.")
    public TaskWithSteps steps(String taskId) {
        return service.getSteps(taskId);
    }

    @MethodHelp(description =
        "Gets information about a step.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "stepId", description = "The position of the step in the step list.") },
        returns = "The step info.")
    public StepState step(String taskId, int stepId) {
        return service.getStep(taskId, stepId, null);
    }

    @MethodHelp(description =
        "Starts a task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") })
    public void start(String taskId) {
        service.start(taskId);
    }

    @MethodHelp(description =
        "Gracefully stops an active task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") })
    public void stop(String taskId) {
        service.stop(taskId);
    }

    @MethodHelp(description =
        "Aborts an active task", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") })
    public void abort(String taskId) {
        service.abort(taskId);
    }

    @MethodHelp(description =
        "Cancels a stopped task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") })
    public void cancel(String taskId) {
        service.cancel(taskId);
    }

    @MethodHelp(description =
        "Archives an executed task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") })
    public void archive(String taskId) {
        service.archive(taskId);
    }
    
    @MethodHelp(description =
        "Skips steps of a task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "stepIds", description = "The ids of the steps to skip.") },
        returns = "The updated task with step information.")
    public TaskWithSteps skip(String taskId, List<Integer> stepIds) {
        return service.skip(taskId, stepIds);
    }

    @MethodHelp(description =
        "Unskips steps of a task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "stepIds", description = "The ids of the steps to skip.") },
        returns = "The updated task with step information.")
    public TaskWithSteps unskip(String taskId, List<Integer> stepIds) {
        return service.unskip(taskId, stepIds);
    }

    @MethodHelp(description =
        "Moves a step within a task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "stepId", description = "The old position of the step."),
            @ParameterHelp(name = "newPosition", description = "The new position of the step.") },
        returns = "The task with step information.")
    public TaskWithSteps moveStep(String taskId, int stepId, int newPosition) {
        return service.moveStep(taskId, stepId, newPosition);
    }

    @MethodHelp(description =
        "Assigns a pending task to another principal.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "owner", description = "The new task owner.") },
        returns = "The task without step information.")
    public TaskState assign(String taskId, String owner) {
        return service.assign(taskId, owner);
    }

    @MethodHelp(description =
        "Gets all task information from the repository's archive in the specified date range.", parameters = {
            @ParameterHelp(name = "beginDate", description = "Begin date from which to return tasks in 'MM/dd/yyyy' format."),
            @ParameterHelp(name = "endDate", description = "End date from which to return tasks in 'MM/dd/yyyy' format.") },
        returns = "A list of tasks without step information.")
    public List<TaskState> query(String begin, String end) {
        return service.query(toLocalDate(begin), toLocalDate(end));
    }

    @MethodHelp(description = "Gets all task information, including steps, from the repository's archive in the specified date range.", parameters = {
        @ParameterHelp(name = "beginDate", description = "Begin date from which to return tasks in 'MM/dd/yyyy' format."),
        @ParameterHelp(name = "endDate", description = "End date from which to return tasks in 'MM/dd/yyyy' format.") },
        returns = "A list of tasks with their enclosed steps.")
    public List<TaskWithSteps> export(String begin, String end) {
        return service.export(toLocalDate(begin), toLocalDate(end));
    }
}
