package com.xebialabs.deployit.cli.api;

import java.util.List;

import org.jboss.resteasy.client.ClientResponseFailure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import com.xebialabs.deployit.booter.remote.HttpClientHolder;
import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.core.api.InternalSecurityProxy;
import com.xebialabs.deployit.core.api.dto.RolePrincipals;
import com.xebialabs.deployit.engine.api.MetadataService;
import com.xebialabs.deployit.engine.api.PermissionService;
import com.xebialabs.deployit.engine.api.RoleService;
import com.xebialabs.deployit.engine.api.ServerService;
import com.xebialabs.deployit.engine.api.UserService;
import com.xebialabs.deployit.engine.api.dto.ServerInfo;
import com.xebialabs.deployit.engine.api.security.Permission;
import com.xebialabs.deployit.engine.api.security.User;

import static com.google.common.collect.Lists.newArrayList;

@CliObject(name = "security")
@ClassHelp(description = "Access to the security settings of Deployit.")
public class SecurityClient extends DocumentedObject {
    private final UserService userService;
    private final PermissionService permissionService;
    private final InternalSecurityProxy internalSecurityProxy;
    private final ServerService serverService;
    private final MetadataService metadataService;
    private final RoleService roleService;

    public SecurityClient() {
        userService = null;
        permissionService = null;
        serverService = null;
        metadataService = null;
        internalSecurityProxy = null;
        roleService = null;
    }

    public SecurityClient(ProxiesInstance proxies) {
        userService = proxies.getUser();
        permissionService = proxies.getPermissions();
        serverService = proxies.getServer();
        metadataService = proxies.getReferenceData();
        internalSecurityProxy = proxies.getInternalSecurity();
        roleService = proxies.getRoleService();
    }

    @MethodHelp(description = "Logout the currently logged in user, can only perform further actions after a login")
    public void logout() {
        try {
            serverService.logout();
        } catch (ClientResponseFailure e) {
            // Assume we are logged out if we get a "401 - Unauthorized"
            // response.
            if (e.getResponse().getStatus() != 401) {
                throw e;
            }
        }
        HttpClientHolder.getInstance().logout();
    }

    @MethodHelp(description = "Login a user", parameters = {
        @ParameterHelp(name = "username", description = "The username"),
        @ParameterHelp(name = "password", description = "The password")
    })
    public void login(String username, String password) {
        logger.info("Logging in as {}", username);
        HttpClientHolder.getInstance().loginAs(username, password);
        try {
            @SuppressWarnings("unused")
            ServerInfo info = serverService.getInfo();
        } catch (Exception e) {
            HttpClientHolder.getInstance().logout();
            throw new IllegalStateException("You're not authorized with these credentials. (" + username + ")", e);
        }
    }

    @MethodHelp(description = "Create a user with the specified name and password", parameters = {
        @ParameterHelp(name = "username", description = "The username"),
        @ParameterHelp(name = "password", description = "The password")
    }, returns = "The created user")
    public User createUser(String username, String password) {
        return createUser(username, password, false);
    }

    public User createUser(final String username, final String password, final boolean admin) {
        User user = new User(username, admin);
        user.setPassword(password);
        return userService.create(username, user);
    }

    @MethodHelp(description = "Read a user so that he/she can be modified.", parameters = {
        @ParameterHelp(name = "username", description = "The username of the user to read")
    }, returns = "The read user")
    public User readUser(String username) {
        return userService.read(username);
    }

    @MethodHelp(description = "Modify the (password of) the user.", parameters = {
        @ParameterHelp(name = "user", description = "The updated user object.")
    })
    public void modifyUser(User user) {
        userService.modifyPassword(user.getUsername(), user);

    }

    @MethodHelp(description = "Delete a user.", parameters = {
        @ParameterHelp(name = "username", description = "the username of the user to be deleted")
    })
    public void deleteUser(String username) {
        userService.delete(username);
    }

    @MethodHelp(description = "Grant a permission to a role", parameters = {
        @ParameterHelp(name = "permission", description = "The permission to grant"),
        @ParameterHelp(name = "roleName", description = "The role")
    })
    public void grant(String permission, String roleName) {
        doGrant(permission, roleName, "global");
    }

    @MethodHelp(description = "Grant a permission to a role on a group of configuration items", parameters = {
        @ParameterHelp(name = "permission", description = "The permission to grant"),
        @ParameterHelp(name = "roleName", description = "The role"),
        @ParameterHelp(name = "configurationItems", description = "A list of configuration items to which the permission should apply")
    })
    public void grant(String permission, String roleName, List<String> configurationItems) {
        List<String> ids = removeTrailingSlashes(configurationItems);
        for (String id : ids) {
            doGrant(permission, roleName, id);
        }
    }

    @MethodHelp(description = "Revoke a permission from a role", parameters = {
        @ParameterHelp(name = "permission", description = "The permission to revoke"),
        @ParameterHelp(name = "roleName", description = "The role")
    })
    public void revoke(String permission, String roleName) {
        doRevoke(permission, roleName, "global");
    }

    @MethodHelp(description = "Revoke a permission from a role on a group of configuration items", parameters = {
        @ParameterHelp(name = "permission", description = "The permission to grant"),
        @ParameterHelp(name = "roleName", description = "The role"),
        @ParameterHelp(name = "configurationItems", description = "A list of configuration items from which the permission should be removed")
    })
    public void revoke(String permission, String roleName, List<String> configurationItems) {
        List<String> ids = removeTrailingSlashes(configurationItems);
        for (String id : ids) {
            doRevoke(permission, roleName, id);
        }
    }

    private void doGrant(String permission, String role, String id) {
        permissionService.grant(permission, id, role);
    }

    private void doRevoke(String permission, String role, String id) {
        permissionService.revoke(permission, id, role);
    }

    @MethodHelp(description = "Check whether a permission is granted to the logged in user on an id.", parameters = {
        @ParameterHelp(name = "permission", description = "The permission to check"),
        @ParameterHelp(name = "principal", description = "The user or group whose permissions are to be retrieved") },
        returns = ("A permissions object containing the user's or group's granted permissions"))
    public boolean hasPermission(String permission, String id) {
        try {
            permissionService.checkMyPermission(permission, id);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    // @MethodHelp(description = "Get permissions granted to a role",
    // parameters = { @ParameterHelp(name = "role", description =
    // "The role whose permissions are to be retrieved") },
    // returns =
    // ("A permissions object containing the role's granted permissions"))
    // public AssignedPermissions getPermissions(final String role) {
    // internalSecurityProxy.
    // return new
    // ResponseExtractor(securityProxy.retrievePermissions(role)).getEntity();
    // }
    //
    // @MethodHelp(description =
    // "Get permissions granted to the currently logged in user")
    // public AssignedPermissions getPermissions() {
    // return new
    // ResponseExtractor(securityProxy.retrievePermissionsForCurrentUser()).getEntity();
    // }
    //

    @MethodHelp(description = "List all the permissions available in Deployit")
    public void printPermissions() {
        System.out.println("Available permissions are:");
        for (Permission permission : metadataService.listPermissions()) {
            System.out.println(permission.getPermissionName());
        }
    }

    @MethodHelp(description = "Create or update a role with assigned principals.", parameters = {
        @ParameterHelp(name = "roleName", description = "The role name"),
        @ParameterHelp(name = "principals", description = "The assigned principals")
    })
    public void assignRole(final String roleName, List<String> principals) {
        if (principals == null || principals.isEmpty()) {
            roleService.create(roleName);
        } else {
            for (String principal : principals) {
                roleService.assign(roleName, principal);
            }
        }
    }

    @MethodHelp(description = "Remove a role.", parameters = {
        @ParameterHelp(name = "roleName", description = "The role name")
    })
    public void removeRole(String roleName) {
        roleService.delete(roleName);
    }

    @MethodHelp(description = "Get all existing roles in Deployit.", returns = "The list of role names")
    public List<String> getRoleNames() {
        return roleService.list();
    }

    public void renameRole(String oldName, String newName) {
        roleService.rename(oldName, newName);
    }

    @MethodHelp(description = "Get the principals assigned to a specific role.", parameters = {
        @ParameterHelp(name = "roleName", description = "The role name")
    }, returns = "A list of principals assigned to the role")
    public List<String> getRoleAssignments(String roleName) {
        List<RolePrincipals> assignments = internalSecurityProxy.readRolePrincipals();
        for (RolePrincipals assignment : assignments) {
            if (assignment.getRole().getName().equals(roleName)) {
                return assignment.getPrincipals();
            }
        }
        return newArrayList();
    }

    private static List<String> removeTrailingSlashes(final List<String> repositoryEntityIds) {
        List<String> cleanRepositoryEntityIds = Lists.<String> newArrayList();
        for (String repositoryEntityId : repositoryEntityIds) {
            cleanRepositoryEntityIds.add(repositoryEntityId.replaceFirst("/*$", ""));
        }
        return cleanRepositoryEntityIds;
    }

    private static final Logger logger = LoggerFactory.getLogger(SecurityClient.class);
}
