package com.xebialabs.deployit.cli;

import java.io.*;
import java.lang.reflect.Constructor;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import javax.script.ScriptException;

import jline.Terminal;
import jline.TerminalFactory;
import jline.console.ConsoleReader;
import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;

import com.google.common.io.Closeables;
import com.xebialabs.deployit.booter.remote.BooterConfig;
import com.xebialabs.deployit.booter.remote.RemoteBooter;
import com.xebialabs.deployit.cli.api.ProxiesInstance;
import com.xebialabs.deployit.cli.help.HelpScanner;

import static com.xebialabs.deployit.cli.CliOptions.EXTENSION_DIR;
import static com.xebialabs.deployit.cli.CliOptions.parse;

public class Cli {

    private static final AtomicReference<Properties> properties = new AtomicReference<Properties>();
    private static ConsoleReader consoleReader;
    private static final FilenameFilter CLI_EXTENSION_FILTER = new FilenameFilter() {
        @Override
        public boolean accept(final File dir, final String name) {
            return name.endsWith(".py") || name.endsWith(".cli");
        }
    };
    private ScriptEngineBuilder scriptEngine;
    private CliOptions options;

    public Cli(CliOptions options) throws Exception {
        this.options = options;
        scriptEngine = new ScriptEngineBuilder();
        initialize();
    }

    private void printBanner() {
        bannerPrint("Welcome to the Deployit Jython CLI!");
        bannerPrint("Type 'help' to learn about the objects you can use to interact with Deployit.");
        bannerPrint("");
    }

    private void initialize() throws Exception {
        BooterConfig.Builder builder = BooterConfig.builder().withHost(options.getHost()).withContext(options.getContext()).withPort(options.getPort());
        if (options.isSecured()) {
            builder.withProtocol(BooterConfig.Protocol.HTTPS);
        }
        createCredentials(builder);
        BooterConfig config = builder.build();
        RemoteBooter.boot(config);
        printBanner();
        ProxiesInstance proxies = createAndRegisterProxies(config);
        registerCliObjects(proxies);
    }

    private ProxiesInstance createAndRegisterProxies(BooterConfig config) {
        ProxiesInstance proxies = new ProxiesInstance(config);
        if (options.isExposeProxies()) {
            System.out.println("Exposing Proxies!");
            scriptEngine.put("proxies", proxies);
        }
        return proxies;
    }

    public static void main(String[] args) throws Exception {
        CliOptions options = parse(args);
        if (options == null)
            return;

        consoleReader = setupConsole();
        new Cli(options).getNewInterpreter().interpret();
    }

    private static ConsoleReader setupConsole() throws IOException {
        final Terminal terminal = TerminalFactory.create();
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                try {
                    terminal.restore();
                } catch (Exception ignored) {
                }
            }
        });
        ConsoleReader cr = new ConsoleReader();
        cr.setExpandEvents(false);
        return cr;
    }

    public Interpreter getNewInterpreter() throws ScriptException, FileNotFoundException {
        final Interpreter interpreter = new Interpreter(consoleReader, scriptEngine, options);
        readExtensions(interpreter);
        return interpreter;
    }

    private void registerCliObjects(final ProxiesInstance proxies) throws Exception {
        final Scannit scannit = new Scannit(Configuration.config().with(new TypeAnnotationScanner()).scan("com.xebialabs"));
        final Set<Class<?>> classes = scannit.getTypesAnnotatedWith(CliObject.class);
        for (Class<?> cliObject : classes) {
            final Constructor<?> constructor = cliObject.getConstructor(ProxiesInstance.class);
            final Object o = constructor.newInstance(proxies);
            final String name = cliObject.getAnnotation(CliObject.class).name();
            scriptEngine.put(name, o);
        }
        if (!options.isQuiet()) {
            HelpScanner.printHelp(classes);
        }
    }

    public void createCredentials(BooterConfig.Builder builder) throws IOException {
        String username;
        String password;
        if (options.isUsernameOnCommandline()) {
            username = options.getUsername();
        } else if (deployitConfigurationFileExists()) {
            username = readFromProperties("cli.username");
        } else {
            username = consoleReader.readLine("username >");
        }

        if (options.isPasswordOnCommandline()) {
            password = options.getPassword();
        } else if (deployitConfigurationFileExists()) {
            password = readFromProperties("cli.password");
        } else {
            password = consoleReader.readLine("password >", '\0');
        }
        builder.withCredentials(username, password);
    }

    private boolean deployitConfigurationFileExists() {
        return options.getConfigurationFile().exists();
    }

    private String readFromProperties(final String key) throws IOException {
        if (properties.get() == null) {
            Properties props = new Properties();
            FileInputStream inStream = new FileInputStream(options.getConfigurationFile());
            try {
                props.load(inStream);
            } finally {
                Closeables.closeQuietly(inStream);
            }
            properties.set(props);
        }

        return properties.get().getProperty(key);
    }

    private void readExtensions(Interpreter interpreter) throws ScriptException, FileNotFoundException {
        final File extensionDir = new File(EXTENSION_DIR);
        if (!extensionDir.exists() || !extensionDir.isDirectory()) {
            System.out.println("No extension directory present.");
            return;
        }

        final File[] files = extensionDir.listFiles(CLI_EXTENSION_FILTER);

        for (File extension : files) {
            bannerPrint("Reading extension: " + extension);
            interpreter.evaluate(new FileReader(extension));
        }
    }

    void bannerPrint(String line) {
        if (!options.isQuiet()) {
            System.out.println(line);
        }
    }
}
