package com.xebialabs.deployit.cli.api.internal;

import static com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyKind.ENUM;
import static com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyKind.LIST_OF_OBJECT;
import static com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyKind.SET_OF_STRING;
import static java.lang.String.format;
import static java.lang.System.out;

import java.util.Arrays;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.cli.api.Proxies;
import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.DescriptorProxy;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDescriptorList;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemMethodDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyKind;

public class DescriptorHelper {
    private static DescriptorHelper instance;

    private DescriptorProxy descriptorProxy;

    private Map<String, String> simpleNameMap = Maps.newHashMap();


    public DescriptorHelper(final Proxies proxies) {
        this.descriptorProxy = proxies.getDescriptors();
        initialize();
    }

    private void initialize() {
        final ConfigurationItemDescriptorList descriptors = new ResponseExtractor(descriptorProxy.list()).getEntity();
        registerSimpleNames(descriptors);
    }

    private void registerSimpleNames(final ConfigurationItemDescriptorList descriptors) {
        for (ConfigurationItemDescriptorDto configurationItemDescriptorDto : descriptors.getDescriptors()) {
            simpleNameMap.put(configurationItemDescriptorDto.getSimpleName(), configurationItemDescriptorDto.getType());
        }
    }

    public static DescriptorHelper getInstance(Proxies proxies) {
        if (instance == null) {
            instance = new DescriptorHelper(proxies);
        }
        return instance;        
    }

    public String getRegisteredType(String shortCiType) {
        return simpleNameMap.get(shortCiType);
    }

    public boolean isTypeRegistered(final String shortCiType) {
        return simpleNameMap.containsKey(shortCiType);
    }

    public static void describe(final String typeName) {
        instance.describeType(typeName);
    }

    private void describeType(final String shortType) {
        final String type = getRegisteredType(shortType);
        if (type == null) {
            out.println();
            out.printf("No ConfigurationItem named \"%s\" found.\n", shortType);
            out.println();
            return;
        }
        ConfigurationItemDescriptorDto ciDescriptor = new ResponseExtractor(descriptorProxy.find(type)).getEntity();

        out.println();
        out.printf("ConfigurationItem \"%s\" (%s):\n", ciDescriptor.getSimpleName(), ciDescriptor.getType());
        out.printf("Description: %s\n", ciDescriptor.getDescription());
        out.println("Control tasks:");
        for (ConfigurationItemMethodDescriptorDto controlTask : ciDescriptor.getControlTasks()) {
            out.printf("\t. %s: %s\n", controlTask.getName(), controlTask.getDescription());
        }
        out.println("Properties:");
        for (ConfigurationItemPropertyDescriptorDto property : ciDescriptor.getPropertyDescriptors()) {
            out.printf("\t%s %s(%s): %s\n", getPropertyModifier(property), property.getName(), getType(property), property.getDescription());
            if (property.getType() == ENUM) {
                out.printf("\t\tValues: %s\n", Arrays.toString(property.getEnumValues()));
            }
        }
        out.println();
        out.println("Properties marked with a '!' are required for discovery.");
        out.println("Properties marked with a '*' are required.");
        out.println();
    }

	private String getPropertyModifier(ConfigurationItemPropertyDescriptorDto property) {
		if (property.isDiscoveryRequired()) {
			return "!";
		}
		return property.isRequired() ? "*" : "-";
	}

    private String getType(final ConfigurationItemPropertyDescriptorDto property) {
        final ConfigurationItemPropertyKind type = property.getType();
        if (type == ConfigurationItemPropertyKind.CI) {
            return property.getPropertyClassname();
        } else if (type == SET_OF_CI) {
            return format("Set<%s>", property.getCollectionMemberClassname());
        } else if (type == SET_OF_STRING) {
            return "Set<String>";
        } else if (type == LIST_OF_OBJECT) {
            return format("List<%s>", property.getCollectionMemberClassname());
        } else {
            return property.getType().name();
        }
    }

	public Set<String> types() {
		return simpleNameMap.keySet();
	}
}
