package com.xebialabs.deployit.cli.api.internal;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.cli.api.Proxies;
import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyDescriptorDto;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;

public class PrintHelper {
	
	private final Set<String> repositoryObjectParentIds = Sets.<String>newHashSet();

	private PrintHelper() {		
	}

	public final static PrintHelper getInstance() {
		return new PrintHelper();
	}

    public void printCi(final RepositoryObject object, final Proxies proxies) {
        printTopLevelCiWithIndent(object, "", proxies);
    }

    private void printTopLevelCiWithIndent(final RepositoryObject object, final String indent, final Proxies proxies) {
        System.out.println(object.getType());
        printCiProperties(object, indent, proxies);
    }

    private void printCiProperties(final RepositoryObject object, final String indent, final Proxies proxies) {
    	repositoryObjectParentIds.add(object.getId());
    	println(indent, "id", object.getId(), false);
        printCalendar(indent, "lastModified", object.getLastModified(), false);
        println(indent, "values", null, true);
        Map<String, ConfigurationItemPropertyDescriptorDto> pds = getPropertyDescriptors(object, proxies);
        printValues(object, (indent + "    "), proxies, pds);
    }

    @SuppressWarnings("unchecked")
    private void printValues(final RepositoryObject object, final String indent, final Proxies proxies, final Map<String, ConfigurationItemPropertyDescriptorDto> pds) {
        final Set<Map.Entry<String, Object>> entries = object.getValues().entrySet();
        final Iterator<Map.Entry<String, Object>> entryIt = entries.iterator();
        while (entryIt.hasNext()) {       
            final Map.Entry<String, Object> entry = entryIt.next();
            final ConfigurationItemPropertyDescriptorDto pd = pds.get(entry.getKey());
            boolean last = !entryIt.hasNext();
            String deepIndent = indent + (entryIt.hasNext() ? "|   " : "    ");
            if (pd == null) {
                println(indent, "(INVALID) " + entry.getKey(), entry.getValue().toString(), last);
            } else {
                switch (pd.getType()) {
                    case BOOLEAN:
                    case INTEGER:
                    case STRING:
                    case ENUM:
                        println(indent, entry.getKey(), entry.getValue().toString(), last);
                        break;
                    case SET_OF_STRING:
                        println(indent, entry.getKey(), null, last);
                        final Collection<String> strings = (Collection<String>) entry.getValue();
                        for (Iterator<String> stringIt = strings.iterator(); stringIt.hasNext();) {
                            String string = stringIt.next();
                            println(deepIndent, string, null, !stringIt.hasNext());
                        }
                        break;
                    case MAP_STRING_STRING:
                    	println(indent, entry.getKey(), null, last);
                    	final Map<String, String> stringMap = (Map<String, String>) entry.getValue();                    	
                    	for (Iterator<String> stringIt = stringMap.keySet().iterator(); stringIt.hasNext();) {
                    		String mapKey = stringIt.next();
                    		println(deepIndent, mapKey, stringMap.get(mapKey), !stringIt.hasNext());
                    	}
                    	break;
                    case LIST_OF_OBJECT:
                        println(indent, entry.getKey(), null, last);
                        printListObjects(deepIndent, (Collection<?>) entry.getValue(), pd);
                        break;
                    case CI:                  
                        final String id = (String) entry.getValue();
                        if (!this.repositoryObjectParentIds.contains(id)) {
                        	final RepositoryObject nested = new ResponseExtractor(proxies.getRepository().read(id)).getEntity();
                        	println(indent, entry.getKey(), nested.getType(), last);
                        	printCiProperties(nested, deepIndent, proxies);
                        } else {
                        	println(indent, entry.getKey(), "REFERENCE => " + id, last);
                        }
                        break;
                    case SET_OF_CI:
                        println(indent, entry.getKey(), null, last);
                        Collection<String> ids = (Collection<String>) entry.getValue();
                        printSetOfCis(deepIndent, ids, proxies);
                        break;
                    case UNSUPPORTED:
                        break;
                }
            }
        }
    }

    private void printSetOfCis(final String indent, final Collection<String> ids, final Proxies proxies) {
        for (Iterator<String> idIt = ids.iterator(); idIt.hasNext();) {
            String nestedid = idIt.next();
            if (!this.repositoryObjectParentIds.contains(nestedid)) {
            	final RepositoryObject nestedCi = new ResponseExtractor(proxies.getRepository().read(nestedid)).getEntity();
            	println(indent, nestedCi.getType(), null, !idIt.hasNext());
            	printCiProperties(nestedCi, indent + (idIt.hasNext() ? "|   " : "    "), proxies);
            } else {
            	println(indent, "REFERENCE => " + nestedid, null, !idIt.hasNext());
            }
        }
    }

    @SuppressWarnings("unchecked")
    private void printListObjects(final String indent, final Collection<?> objects, final ConfigurationItemPropertyDescriptorDto dto) {
        for (Iterator<?> objectIt = objects.iterator(); objectIt.hasNext();) {
            final Map<String, String> object = (Map<String, String>) objectIt.next();
            println(indent, dto.getCollectionMemberClassname(), null, !objectIt.hasNext());
            for(Iterator<Map.Entry<String, String>> objEntryIt = object.entrySet().iterator(); objEntryIt.hasNext();) {
                final Map.Entry<String, String> entry = objEntryIt.next();
                println(indent + (objectIt.hasNext() ? "|   " : "    "), entry.getKey(), entry.getValue(), !objEntryIt.hasNext());
            }
        }
    }

    private Map<String, ConfigurationItemPropertyDescriptorDto> getPropertyDescriptors(final RepositoryObject object, final Proxies proxies) {
        final ConfigurationItemDescriptorDto descriptor = new ResponseExtractor(proxies.getDescriptors().find(object.getType())).getEntity();
        final List<ConfigurationItemPropertyDescriptorDto> propertyDescriptors = descriptor.getPropertyDescriptors();
        Map<String, ConfigurationItemPropertyDescriptorDto> pds = Maps.newHashMap();
        for (ConfigurationItemPropertyDescriptorDto propertyDescriptor : propertyDescriptors) {
            pds.put(propertyDescriptor.getName(), propertyDescriptor);
        }
        return pds;
    }

    private void println(String indent, String key, String value, boolean last) {
        System.out.println(indent + (last ? "\\-- " : "+-- ") + key + (value != null ? ": " + value : ""));
    }

    private void printCalendar(final String indent, final String key, final Calendar value, final boolean last) {
        if (value != null) {
            println(indent, key, new SimpleDateFormat("yyyy-MM-dd hh:MM:ss.SSS").format(value.getTime()), last);
        } else {
            println(indent, key, null, last);
        }
    }
}
