package com.xebialabs.deployit.cli.api;

import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.core.api.dto.Artifact;
import com.xebialabs.deployit.core.api.dto.ArtifactAndData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.api.internal.DescriptorHelper;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.core.api.dto.Archetype;
import com.xebialabs.deployit.core.api.dto.ConfigurationItem;

import static com.google.common.collect.Maps.newHashMap;

@CliObject(name = "factory")
@ClassHelp(description = "Helper that can construct archetypes, CIs and artifacts")
public class ObjectFactory extends DocumentedObject {
    private DescriptorHelper descriptors;

    public ObjectFactory() {
    }

    public ObjectFactory(Proxies proxies) {
        descriptors = DescriptorHelper.getInstance(proxies);
    }

    @MethodHelp(description = "Construct a CI of a specified type", parameters = {
        @ParameterHelp(name = "ciType", description = "The (shortname) type of the CI")
    })
    public ConfigurationItem configurationItem(String ciType) {
        return configurationItem(ciType, Maps.<String, Object>newHashMap());
    }

    @MethodHelp(description = "Construct a CI of a specified type with specified values", parameters = {
        @ParameterHelp(name = "ciType", description = "The (shortname) type of the CI"),
        @ParameterHelp(name = "values", description = "The values of the CI in a map.")
    })
    public ConfigurationItem configurationItem(final String ciType, final Map<String, Object> values) {
        if (!checkSimpleNameMap(ciType)) return null;
        final String type = descriptors.getRegisteredType(ciType);
        final ConfigurationItem configurationItem = new ConfigurationItem(type);
        // Need to wrap the incoming map into a hashmap, as it could be a PyDict.
        configurationItem.setValues(Maps.<String, Object>newHashMap(values));
        return configurationItem;
    }

    @MethodHelp(description = "Construct an  archetype for a CI of a specified type", parameters = {
        @ParameterHelp(name = "ciType", description = "The (shortname) type of the CI")
    })
    public Archetype archetype(String ciType) {
        return archetype(ciType, Maps.<String, Object>newHashMap());
    }

    @MethodHelp(description = "Construct an  archetype for a CI of a specified type with specified base values", parameters = {
        @ParameterHelp(name = "ciType", description = "The (shortname) type that CIs based on this archetype will be"),
        @ParameterHelp(name = "values", description = "The base values for the archetype in a map.")
    })
    public Archetype archetype(String ciType, Map<String, Object> values) {
        if (!checkSimpleNameMap(ciType)) return null;
        final String type = descriptors.getRegisteredType(ciType);
        final Archetype archetype = new Archetype(type);
        // Need to wrap the incoming map into a hashmap, as it could be a PyDict.
        archetype.setValues(Maps.<String, Object>newHashMap(values));
        return archetype;
    }

	@MethodHelp(description = "Construct an artifact of a specific type with associated data", parameters = {
		@ParameterHelp(name = "ciType", description = "The (shortname) type of this artifact"),
		@ParameterHelp(name = "values", description = "The values for this artifact in a map."),
		@ParameterHelp(name = "data", description = "The data of this artifact that is to be uploaded")
	}, returns = "An Artifact with data that can be stored in Deployit.")
	public ArtifactAndData artifact(String ciType, Map<String, Object> values, byte[] data) {
		if (!checkSimpleNameMap(ciType)) return null;
		final String type = descriptors.getRegisteredType(ciType);
		final Artifact artifact = new Artifact(type);
		artifact.setValues(newHashMap(values));
		return new ArtifactAndData(artifact, data);
	}
	
    private boolean checkSimpleNameMap(final String ciType) {
        boolean contains = descriptors.isTypeRegistered(ciType);
        if (!contains) {
            logger.error("Could not find CI with name {}", ciType);
        }
        return contains;
    }
	@MethodHelp(description = "Prints all registered configuration item types.")
	public void types() {
		final Set<String> typeSet = descriptors.types();
		System.out.printf("The registered configuration item types are:\n%s\n\n", typeSet);
	}

    private static final Logger logger = LoggerFactory.getLogger(ObjectFactory.class);
}
