package com.xebialabs.deployit.cli.api.internal;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.cli.api.Proxies;
import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyDescriptorDto;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;

import java.text.SimpleDateFormat;
import java.util.*;

public class PrintHelper {

    public static void printCi(RepositoryObject object, Proxies proxies) {
        printTopLevelCiWithIndent(object, "", proxies);
    }

    private static void printTopLevelCiWithIndent(final RepositoryObject object, String indent, Proxies proxies) {
        System.out.println(object.getConfigurationItemTypeName());
        printCiProperties(object, indent, proxies);
    }

    private static void printCiProperties(final RepositoryObject object, String indent, final Proxies proxies) {
        println(indent, "id", object.getId(), false);
        printCalendar(indent, "lastModified", object.getLastModified(), false);
        println(indent, "values", null, true);
        Map<String, ConfigurationItemPropertyDescriptorDto> pds = getPropertyDescriptors(object, proxies);
        indent = indent + "    ";

        printValues(object, indent, proxies, pds);
    }

    private static void printValues(final RepositoryObject object, final String indent, final Proxies proxies, final Map<String, ConfigurationItemPropertyDescriptorDto> pds) {
        final Set<Map.Entry<String, Object>> entries = object.getValues().entrySet();
        for (Iterator<Map.Entry<String, Object>> entryIt = entries.iterator(); entryIt.hasNext();) {
            final Map.Entry<String, Object> entry = entryIt.next();
            final ConfigurationItemPropertyDescriptorDto dto = pds.get(entry.getKey());
            boolean last = !entryIt.hasNext();
            String deepIndent = indent + (entryIt.hasNext() ? "|   " : "    ");
            if (dto == null) {
                println(indent, "(INVALID) " + entry.getKey(), entry.getValue().toString(), last);
            } else {
                switch (dto.getType()) {
                    case BOOLEAN:
                    case INTEGER:
                    case STRING:
                    case ENUM:
                        println(indent, entry.getKey(), entry.getValue().toString(), last);
                        break;
                    case SET_OF_STRINGS:
                        println(indent, entry.getKey(), null, last);
                        final Collection<String> strings = (Collection<String>) entry.getValue();
                        for (Iterator<String> stringIt = strings.iterator(); stringIt.hasNext();) {
                            String string = stringIt.next();
                            println(deepIndent, string, null, !stringIt.hasNext());
                        }
                        break;
                    case LIST_OF_OBJECTS:
                        println(indent, entry.getKey(), null, last);
                        printListObjects(deepIndent, (Collection<?>) entry.getValue(), dto);
                        break;
                    case CI:
                        final String id = (String) entry.getValue();
                        final RepositoryObject nested = new ResponseExtractor(proxies.getRepository().read(id)).getEntity();
                        println(indent, entry.getKey(), nested.getConfigurationItemTypeName(), last);
                        printCiProperties(nested, deepIndent, proxies);
                        break;
                    case SET_OF_CIS:
                        println(indent, entry.getKey(), null, last);
                        Collection<String> ids = (Collection<String>) entry.getValue();
                        printSetOfCis(deepIndent, ids, proxies);
                        break;
                    case UNSUPPORTED:
                        break;
                }
            }
        }
    }

    private static void printSetOfCis(final String indent, final Collection<String> ids, final Proxies proxies) {
        for (Iterator<String> idIt = ids.iterator(); idIt.hasNext();) {
            String nestedid = idIt.next();
            final RepositoryObject nestedCi = new ResponseExtractor(proxies.getRepository().read(nestedid)).getEntity();
            println(indent, nestedCi.getConfigurationItemTypeName(), null, !idIt.hasNext());
            printCiProperties(nestedCi, indent + (idIt.hasNext() ? "|   " : "    "), proxies);
        }
    }

    private static void printListObjects(final String indent, final Collection<?> objects, final ConfigurationItemPropertyDescriptorDto dto) {
        for (Iterator objectIt = objects.iterator(); objectIt.hasNext();) {
            final Map<String, String> object = (Map<String, String>) objectIt.next();
            println(indent, dto.getCollectionMemberClassname(), null, !objectIt.hasNext());
            for(Iterator<Map.Entry<String, String>> objEntryIt = object.entrySet().iterator(); objEntryIt.hasNext();) {
                final Map.Entry<String, String> entry = objEntryIt.next();
                println(indent + (objectIt.hasNext() ? "|   " : "    "), entry.getKey(), entry.getValue(), !objEntryIt.hasNext());
            }
        }
    }

    private static Map<String, ConfigurationItemPropertyDescriptorDto> getPropertyDescriptors(final RepositoryObject object, final Proxies proxies) {
        final ConfigurationItemDescriptorDto descriptor = new ResponseExtractor(proxies.getDescriptors().find(object.getConfigurationItemTypeName())).getEntity();
        final List<ConfigurationItemPropertyDescriptorDto> propertyDescriptors = descriptor.getPropertyDescriptors();
        Map<String, ConfigurationItemPropertyDescriptorDto> pds = Maps.newHashMap();
        for (ConfigurationItemPropertyDescriptorDto propertyDescriptor : propertyDescriptors) {
            pds.put(propertyDescriptor.getName(), propertyDescriptor);
        }
        return pds;
    }

    private static void println(String indent, String key, String value, boolean last) {
        System.out.println(indent + (last ? "\\-- " : "+-- ") + key + (value != null ? ": " + value : ""));
    }

    private static void printCalendar(final String indent, final String key, final Calendar value, final boolean last) {
        if (value != null) {
            println(indent, key, new SimpleDateFormat("yyyy-MM-dd hh:MM:ss.SSS").format(value.getTime()), last);
        } else {
            println(indent, key, null, last);
        }
    }
}
