param(
	[string][parameter (mandatory=$true)][ValidateNotNullOrEmpty()]$connectedServiceName,
	[string][parameter (mandatory=$true)][ValidateNotNullOrEmpty()]$manifestPath,
	[string]$publish,
	[string]$version,
	[string]$deploy,
	[string]$targetEnvironment,
	[string]$rollback,
    [string]$outputFolder,
    [string]$rootFolder,
    [string]$versionNumber
)

Write-Verbose "Entering script task.ps1"

Write-Verbose "connectedServiceName = $connectedServiceName"
Write-Verbose "manifestPath = $manifestPath"
Write-Verbose "publish = $publish"
Write-Verbose "version = $version"
Write-Verbose "deploy = $deploy"
Write-Verbose "targetEnvironment = $targetEnvironment"
Write-Verbose "rollback = $rollback"
Write-Verbose "outputFolder = $outputFolder"
Write-Verbose "rootFolder = $rootFolder"
Write-Verbose "versionNumber = $versionNumber"

$ErrorActionPreference = "Stop"

Import-Module $PSScriptRoot\xld-import.psm1
Import-Module $PSScriptRoot\xld-package.psm1
Import-Module $PSScriptRoot\xld-verify.psm1

Import-Module "Microsoft.TeamFoundation.DistributedTask.Task.Common"

# Sanitize
if ($publish)
{
	Write-Verbose "Trying to convert publish parameter to boolean."
	$shouldPublish = [System.Convert]::ToBoolean($publish)
	Write-Verbose "shouldPublish = $shouldPublish"
}

if ($version)
{
	Write-Verbose "Trying to convert version parameter to boolean."
	$shouldVersion = [System.Convert]::ToBoolean($version)
	Write-Verbose "shouldVersion = $shouldVersion"
}

if ($deploy)
{
	Write-Verbose "Trying to convert deploy parameter to boolean."
	$shouldDeploy = [System.Convert]::ToBoolean($deploy)
	Write-Verbose "shouldDeploy = $shouldDeploy"
}

if ($rollback)
{
	Write-Verbose "Trying to convert rollback parameter to boolean."
	$shouldRollback = [System.Convert]::ToBoolean($rollback)
	Write-Verbose "shouldRollback = $shouldRollback"
}

# Check if path to manifest file is valid
Write-Verbose "Validating manifest path..."
if (-not (Test-Path $manifestPath -PathType Leaf)) 
{
	throw "Manifest file not found. $manifestPath is not a valid path."
}
Write-Verbose "Manifest file found at $manifestPath"

# Verify output folder
if ($outputFolder)
{
    # if folder doesn't exist or it's not valid
    if (-not (Test-Path $outputFolder -PathType Container -IsValid))
    {
        throw "Provided output folder is not a valid path."
    }
}
else
{
    $outputFolder = $env:BUILD_STAGINGDIRECTORY
}

# Verify root folder
if ($rootFolder)
{
    # if folder doesn't exist or it's not valid
    if (-not (Test-Path $rootFolder -PathType Container -IsValid))
    {
        throw "Provided root folder is not a valid path."
    }
}
else
{
    $rootFolder = $env:BUILD_STAGINGDIRECTORY
}

# Validate Service Endpoint
$serviceEndpoint = Get-EndpointData $connectedServiceName

# Create PSCredential object
$credential = New-PSCredential $serviceEndpoint.Authorization.Parameters.UserName $serviceEndpoint.Authorization.Parameters.Password
$serverUrl = Test-EndpointBaseUrl $serviceEndpoint.Url

# Add URL and credentials to default parameters so that we don't need
# to specify them over and over for this session.
$PSDefaultParameterValues.Add("*:EndpointUrl", $serverUrl)
$PSDefaultParameterValues.Add("*:Credential", $credential)

############################################################################################

# Check server state and validate the address
Write-Output "Checking XL Deploy server state..."
if ((Get-ServerState) -ne "RUNNING")
{
	throw "XL Deploy server not in running state."
}
Write-Output "XL Deploy server is running."

# If deployment options was chosen check uptime if the specified environment is valid
if ($shouldDeploy)
{
	if (-not $targetEnvironment.StartsWith("Environments/", "InvariantCultureIgnoreCase"))
	{
		$targetEnvironment = "Environments/$targetEnvironment"
	}

	if (-not (Test-EnvironmentExists $targetEnvironment)) 
	{
		throw "Specified environment $targetEnvironment doesn't exists."
	}
}

############################################################################################

# Substitute version place holder with version number in manifest file
if ($shouldVersion)
{
	Write-Output ("Setting manifest version to {0}..." -f $versionNumber)
    
    if (-not $versionNumber)
    {
        $versionNumber = $env:BUILD_BUILDNUMBER
    }
    
	$null = Set-Version $manifestPath $versionNumber
    
	Write-Output "Package version set to $versionNumber."
}

############################################################################################

# Create DAR package
Write-Output "Creating DAR package..."
$packageFullPath = New-DarPackage $manifestPath $outputFolder $rootFolder
Write-Output "DAR package created at $packageFullPath."

############################################################################################

# Upload package to XL Deploy
Write-Output "Starting package upload..."
$deploymentPackageId = Send-Package $packageFullPath
Write-Output "Package successfully uploaded. Package ID is $deploymentPackageId"

############################################################################################

# Publish Build artifacts to the TFS server
if ($shouldPublish)
{
    $folderName = "XL Deploy - Deployment Archives"
	Write-Output "Publishing build artifact..."
    Write-Output ("##vso[artifact.upload containerfolder=$folderName;artifactname=$folderName;]$packageFullPath")
	Write-Output "Build artifact published successfully."
}

############################################################################################

# Trigger the deployment if target environment is specified
if ($shouldDeploy)
{
	Import-Module $PSScriptRoot\xld-deploy.psm1

	Write-Output "Starting deployment to $targetEnvironment."
	# create new deployment task
	$deploymentTaskId = New-DeploymentTask $deploymentPackageId $targetEnvironment

	Start-Task $deploymentTaskId

	$taskOutcome = Get-TaskOutcome $deploymentTaskId

	if ($taskOutcome -eq "EXECUTED" -or $taskOutcome -eq "DONE")
	{
		# archive
		Complete-Task $deploymentTaskId
		Write-Output "Successfully deployed to $targetEnvironment."
	}
	else
	{
		if (!$shouldRollback) 
		{
			throw "Deployment failed."
		}

		Write-Warning "Deployment failed."
        Write-Output ("##vso[task.complete result=SucceededWithIssues;]Deployment failed.")
        
		Write-Output "Starting rollback."

		# rollback
		$rollbackTaskId = New-RollbackTask $deploymentTaskId

		Start-Task $rollbackTaskId

		$rollbackTaskOutcome = Get-TaskOutcome $rollbackTaskId

		if ($rollbackTaskOutcome -eq "EXECUTED" -or $rollbackTaskOutcome -eq "DONE")
		{
			# archive
			Complete-Task $rollbackTaskId
			Write-Output "Rollback executed successfully."
		}
		else
		{
			throw "Rollback failed." 
		}
	}
}

Write-Verbose "Leaving script task.ps1"