<############################################################################################ 
    Return information about current server state (is it RUNNING or in MAINTENANCE mode).
############################################################################################>
function Get-ServerState()
{
    [CmdletBinding()]
    param
    (
        [string][parameter(Mandatory = $true)]$EndpointUrl,
        [System.Management.Automation.PSCredential][parameter(Mandatory = $true)]$Credential
    )
    BEGIN { }
    PROCESS
    {
        $response = Invoke-RestMethod $EndpointUrl/server/state -Credential $Credential

        return $response.'server-state'.'current-mode'
    }
    END { }
}

<############################################################################################ 
    Checks if the given environment exists.
############################################################################################>
function Test-EnvironmentExists()
{
    [CmdletBinding()]
    param
    (
        [string][parameter(Mandatory = $true)]$targetEnvironment,
        [string][parameter(Mandatory = $true)]$EndpointUrl,
        [System.Management.Automation.PSCredential][parameter(Mandatory = $true)]$Credential
    )
    BEGIN
    {
        if ($targetEnvironment.StartsWith("Environments/", "InvariantCultureIgnoreCase"))
        {
            $targetEnvironment = $targetEnvironment.Substring(13, $targetEnvironment.Length - 13)
        }

        $targetEnvironment = Get-EncodedPathPart($targetEnvironment)
    }
    PROCESS
    {
        $response = Invoke-RestMethod $EndpointUrl/repository/exists/Environments/$targetEnvironment -Credential $Credential

        return [boolean]::Parse($response.boolean)
    }
    END { }
}

<############################################################################################ 
    Verifies that the endpoint URL is well formatted and a valid URL.
############################################################################################>
function Test-EndpointBaseUrl()
{
	[CmdletBinding()]
	param
	(
		[Uri][parameter(Mandatory = $true)]$Endpoint
	)
	BEGIN
	{
		Write-Verbose "Endpoint = $Endpoint"
	}
	PROCESS 
	{
		#$xldServer = $serviceEndpoint.Url.AbsoluteUri.TrimEnd('/')
		$xldServer = $Endpoint.AbsoluteUri.TrimEnd('/')

		if (-not $xldServer.EndsWith("deployit", "InvariantCultureIgnoreCase"))
		{
			$xldServer = "$xldServer/deployit"
		}

		# takes in consideration both http and https protocol
		if (-not $xldServer.StartsWith("http", "InvariantCultureIgnoreCase"))
		{
			$xldServer = "http://$xldServer"
		}

		$uri = $xldServer -as [System.Uri] 
		if (-not ($null -ne $uri.AbsoluteURI -and $uri.Scheme -match '[http|https]'))
		{
			throw "Provided endpoint address is not a valid URL."
		}

		return $uri
	}
	END { }
}

<############################################################################################ 
	Retrieves the URL, username and password from the specified generic endpoint.
	Only UserNamePassword authentication scheme is supported for XL Deploy.
############################################################################################>
function Get-EndpointData()
{
	[CmdletBinding()]
	param
	(
		[string][parameter(Mandatory = $true)][ValidateNotNullOrEmpty()]$ConnectedServiceName
	)
	BEGIN
	{
		Write-Verbose "ConnectedServiceName = $ConnectedServiceName"
	}
	PROCESS
	{
		$serviceEndpoint = Get-ServiceEndpoint -Context $distributedTaskContext -Name $ConnectedServiceName

		if (!$serviceEndpoint)
		{
			throw "A Connected Service with name '$ConnectedServiceName' could not be found.  Ensure that this Connected Service was successfully provisioned using the services tab in the Admin UI."
		}

		$authScheme = $serviceEndpoint.Authorization.Scheme
		if ($authScheme -ne 'UserNamePassword')
		{
			throw "The authorization scheme $authScheme is not supported by Xl Deploy server."
		}

		Write-Verbose "serverUrl = $($serviceEndpoint.Url)"

		return $serviceEndpoint
	}
	END { }
}


<############################################################################################ 
    Given the username and password strings, create a valid PSCredential object.
############################################################################################>
function New-PSCredential()
{
	[CmdletBinding()]
	param
	(
		[string][parameter(Mandatory = $true)]$Username,
		[string][parameter(Mandatory = $true)]$Password
	)
	BEGIN
	{
		#Write-Verbose "Username = $Username"
        #Write-Verbose "Password = $Password"
	}
	PROCESS
	{
		$securePassword = ConvertTo-SecureString -String $Password -asPlainText -Force
		$credential = New-Object -TypeName System.Management.Automation.PSCredential -ArgumentList $Username, $securePassword

		return $credential
	}
	END { }
}

<############################################################################################ 
    Encodes each part of the path separately.
############################################################################################>
function Get-EncodedPathPart()
{
    [CmdletBinding()]
    param
    (
        [string][parameter(Mandatory = $true)]$PartialPath
    )
    BEGIN { }
    PROCESS
    {
        return ($PartialPath -split "/" | ForEach-Object { [System.Uri]::EscapeDataString($_) }) -join "/"
    }
    END { }
}

<############################################################################################ 
    Verifies if the given is valid path and if the file is a valid manifest file.
############################################################################################>
function Test-ManifestFile()
{
    [CmdletBinding()]
    [OutputType([Boolean])]
    param
    (
        [string][parameter(Mandatory = $true)]$ManifestPath
    )
    BEGIN { }
    PROCESS
    {
        if (-not (Test-Path $ManifestPath -PathType Leaf)) 
        {
	        throw "Manifest file not found. $ManifestPath is not a valid path."
        }

        try
        {
            [xml]$manifest = Get-Content $ManifestPath
        }
        catch [Exception]
        {
            throw "$ManifestPath is not a valid xml document."
        }

        $deploymentPackageElement = $manifest.'udm.DeploymentPackage'

        if (-not $deploymentPackageElement)
        {
            throw "$ManifestPath is not a valid manifest xml document."
        }
        
        return $true
    }
    END { }
}

function Set-Version()
{
    [CmdletBinding(SupportsShouldProcess=$true)]
    param
    (
        [string][parameter(Mandatory = $true)]$ManifestPath,
        [string][parameter(Mandatory = $true)]$Version        
    )
    BEGIN
    {
        if (-not (Test-Path $ManifestPath -PathType Leaf)) 
        {
	        throw "Manifest file not found. $ManifestPath is not a valid path."
        }
        
        if (-not $Version)
        {
            throw "Version number is not specified."
        }
    }
    PROCESS
    {
        if($pscmdlet.ShouldProcess($ManifestPath))
        {
            try
            {
                [xml]$manifest = Get-Content $ManifestPath
                
                $deploymentPackageElement = $manifest.'udm.DeploymentPackage'

                if (-not $deploymentPackageElement)
                {
                    throw "$ManifestPath is not a valid manifest xml document."
                }
                
                $deploymentPackageElement.version = $Version
                
                $manifest.Save($ManifestPath)
            }
            catch [Exception]
            {
                throw "Can't set verion on $ManifestPath."
            }
        }
    }
    END { } 
}

Export-ModuleMember -function *-*