<############################################################################################ 
    Creates a DAR package based on the information in the specified in manifest file.
############################################################################################>
function New-DarPackage()
{
    [CmdletBinding()]
    param
    (
        [string][parameter(Mandatory = $true, ValueFromPipeline = $true)]$manifestPath,
        [string][parameter(Mandatory = $true, ValueFromPipeline = $false)]$rootPath,
        [string][parameter(Mandatory = $true, ValueFromPipeline = $false)]$outputPath        
    )
    BEGIN 
    {
        Write-Verbose "manifestPath = $manifestPath"
        Write-Verbose "rootPath = $rootPath"

        if (-not (Test-Path $rootPath -PathType Container))
        {
            $errorMessage = ("Provided root path {0} is not valid." -f $rootPath)
            ThrowTerminatingErrorHelper 'RootPath' $errorMessage ([System.Management.Automation.ErrorCategory]::InvalidArgument) $rootPath
        }
    }
    PROCESS 
    {
        # XL Deploy expects that the manifest file has a given name
        # of 'deployit-manifest.xml'
        $manifestFileFullPath = "$rootPath\deployit-manifest.xml"

        if (Test-Path $manifestFileFullPath -PathType Leaf)
        {
            ### ? What if already there ?
        }
        else
        {
            # make sure that manifest file is accessible and it
            # is named correctly by coping it to the root path with fixed name.
            Copy-Item -Path $manifestPath -Destination $manifestFileFullPath -force
        }

        $filesToInclude = @("deployit-manifest.xml")
        $filesToInclude += Get-PathsFromManifest $manifestFileFullPath

        $filesToInclude | Write-Verbose

        if (Test-Path $manifestFileFullPath -PathType Leaf)
        {
            $packageFullPath = Join-Path $outputPath "Package.dar"    
        }
        else
        {
            # file already exists
            $errorMessage = ("A DAR package already exists at {0}." -f $packageFullPath)
            ThrowTerminatingErrorHelper 'PackageExists' $errorMessage ([System.Management.Automation.ErrorCategory]::ResourceExists) $packageFullPath
        }
        
        if (Test-Path $packageFullPath -PathType Leaf)
        {
            # file already exists
            $errorMessage = ("A DAR package already exists at {0}." -f $packageFullPath)
            ThrowTerminatingErrorHelper 'PackageExists' $errorMessage ([System.Management.Automation.ErrorCategory]::ResourceExists) $packageFullPath
        }
                
        $filesToInclude | Compress-Package -DestinationPath $packageFullPath -RootPath $rootPath
        
        return $packageFullPath
    }
    END { }
}

<############################################################################################ 
    Parses the manifest file in search for file attribute and returns an array of
    relative path values.
############################################################################################>
function Get-PathsFromManifest()
{
    [CmdletBinding()]
    param
    (
        [parameter(Mandatory = $true, ValueFromPipeline = $true)]
        [string]$manifestFileFullPath
    )
    BEGIN
    {
        Write-Verbose "manifestFileFullPath = $manifestFileFullPath"
    }
    PROCESS
    {
        $xml = [xml](Get-Content $manifestFileFullPath)

        return $xml.'udm.DeploymentPackage'.deployables `
            | Select-Object -ExpandProperty ChildNodes `
            | Where-Object  {$_.GetType() -eq [System.Xml.XmlElement] -and $_.HasAttribute("file")} `
            | Select-Object -ExpandProperty file | ForEach-Object { $_ -replace "/", "\"}
    }
    END { }
}

<############################################################################################ 
		# The Compress-Package cmdlet can be used to compress one or more files/directories.
############################################################################################>
function Compress-Package
{
	[CmdletBinding(DefaultParameterSetName='Path', SupportsShouldProcess=$true)]
	param
	(
		[parameter (mandatory=$true, Position=0, ValueFromPipeline=$true)]
		[ValidateNotNullOrEmpty()]
		[string[]] $Path,

		[parameter (mandatory=$true, Position=1)]
		[ValidateNotNullOrEmpty()]
		[string] $RootPath,

		[parameter (mandatory=$true, Position=2)]
		[ValidateNotNullOrEmpty()]
		[string] $DestinationPath
	)

	BEGIN 
	{         
		$inputPaths = @()
		$destinationParentDir = [system.IO.Path]::GetDirectoryName($DestinationPath)
		if($null -eq $destinationParentDir)
		{
			$errorMessage = ("The destination path '{0}' does not contain a valid archive file name." -f $DestinationPath)
			ThrowTerminatingErrorHelper 'InvalidArchiveFilePath' $errorMessage ([System.Management.Automation.ErrorCategory]::InvalidArgument) $DestinationPath
		}

		if($destinationParentDir -eq [string]::Empty)
		{
			$destinationParentDir = '.'
		}

		$achiveFileName = [system.IO.Path]::GetFileName($DestinationPath)
		$destinationParentDir = GetResolvedPathHelper $destinationParentDir $null $PSCmdlet
		
		if($destinationParentDir.Count -gt 1)
		{
			$errorMessage = ("The archive file path '{0}' specified as input to the {1} parameter is resolving to multiple file system paths. Provide a unique path to the {2} parameter where the archive file has to be created." -f $DestinationPath, 'DestinationPath', 'DestinationPath')
			ThrowTerminatingErrorHelper 'InvalidArchiveFilePath' $errorMessage ([System.Management.Automation.ErrorCategory]::InvalidArgument) $DestinationPath
		}

		IsValidFileSystemPath $destinationParentDir | Out-Null
		$DestinationPath = Join-Path -Path $destinationParentDir -ChildPath $achiveFileName

		# GetExtension API does not validate for the actual existence of the path.
		$extension = [system.IO.Path]::GetExtension($DestinationPath)

		# If user does not specify .Dar extension, we append it.
		if($extension -eq [string]::Empty)
		{
			$DestinationPathWithOutExtension = $DestinationPath
			$DestinationPath = $DestinationPathWithOutExtension + '.dar'
			$appendArchiveFileExtensionMessage = ("The archive file path '{0}' supplied to the DestinationPath parameter does not include .zip extension. Hence .zip is appended to the supplied DestinationPath path and the archive file would be created at '{1}'." -f $DestinationPathWithOutExtension, $DestinationPath)
			Write-Verbose $appendArchiveFileExtensionMessage
		}

		$archiveFileExist = Test-Path -Path $DestinationPath -PathType Leaf

		if($archiveFileExist)
		{
			$errorMessage = ("The archive file {0} already exists. Use the -Update parameter to update the existing archive file or use the -Force parameter to overwrite the existing archive file." -f $DestinationPath)
			ThrowTerminatingErrorHelper 'ArchiveFileExists' $errorMessage ([System.Management.Automation.ErrorCategory]::InvalidArgument) $DestinationPath
		}

		if(Test-Path -LiteralPath $RootPath -PathType Container) 
		{
			if(!$RootPath.EndsWith('\')) 
			{
				$RootPath += '\'
			}
		}
		else 
		{
			$errorMessage = ("The provided root path '{0}' is not a valid folder or it doesn't exist." -f $RootPath)
			ThrowTerminatingErrorHelper 'InvalidRootPath' $errorMessage ([System.Management.Automation.ErrorCategory]::InvalidArgument) $RootPath
		}
	}
	PROCESS 
	{
		$inputPaths += $Path
	}
	END 
	{        
		ValidateDuplicateFileSystemPath $PsCmdlet.ParameterSetName $inputPaths
		$resolvedPaths = GetResolvedPathHelper $inputPaths $RootPath $PSCmdlet
		IsValidFileSystemPath $resolvedPaths | Out-Null
	
		$sourcePath = $resolvedPaths;

		# CSVHelper: This is a helper function used to append comma after each path specified by
		# the $sourcePath array. The comma separated paths are displayed in the -WhatIf message.
		$sourcePathInCsvFormat = CSVHelper $sourcePath
		if($pscmdlet.ShouldProcess($sourcePathInCsvFormat))
		{
			try
			{
				# StopProcessing is not available in Script cmdlets. However the pipeline execution
				# is terminated when ever 'CTRL + C' is entered by user to terminate the cmdlet execution.
				# The finally block is executed whenever pipeline is terminated. 
				# $isArchiveFileProcessingComplete variable is used to track if 'CTRL + C' is entered by the
				# user. 
				$isArchiveFileProcessingComplete = $false

				$numberOfItemsArchived = CompressArchiveHelper $sourcePath $RootPath $DestinationPath $Update

				$isArchiveFileProcessingComplete = $true
			}
			finally
			{
				# The $isArchiveFileProcessingComplete would be set to $false if user has typed 'CTRL + C' to 
				# terminate the cmdlet execution or if an unhandled exception is thrown.
				# $numberOfItemsArchived contains the count of number of files or directories add to the archive file.
				# If the newly created archive file is empty then we delete it as its not usable.
				if(($isArchiveFileProcessingComplete -eq $false) -or 
				($numberOfItemsArchived -eq 0))
				{
					$DeleteArchiveFileMessage = ("The partially created archive file '{0}' is deleted as it is not usable." -f $DestinationPath)
					Write-Verbose $DeleteArchiveFileMessage

					# delete the partial archive file created.
					Remove-Item "$DestinationPath" -Force -Recurse
				}
			}
		}
	}
}

<############################################################################################ 
		# GetResolvedPathHelper: This is a helper function used to resolve the user specified Path.
		# The path can either be absolute or relative path.
############################################################################################>
function GetResolvedPathHelper 
{
	param 
	(
		[string[]] $path,
		[string]$root,
		[System.Management.Automation.PSCmdlet]
		$callerPSCmdlet
	)
	
	$resolvedPaths =@()

	# null and empty check are already done on Path parameter at the cmdlet layer.
	foreach($currentPath in $path)
	{
		$fullPath = ''

		try
		{
			if($root)
			{
				$fullPath = Join-Path $root -ChildPath $currentPath
			}
			else 
			{
				$fullPath = $currentPath
			}

			$currentResolvedPaths = Resolve-Path -LiteralPath $fullPath -ErrorAction Stop
		}
		catch
		{
			$errorMessage = ("The path '{0}' either does not exist or is not a valid file system path." -f $fullPath)
			$exception = New-Object System.InvalidOperationException $errorMessage, $_.Exception
			$errorRecord = CreateErrorRecordHelper 'ArchiveCmdletPathNotFound' $null ([System.Management.Automation.ErrorCategory]::InvalidArgument) $exception $fullPath
			$callerPSCmdlet.ThrowTerminatingError($errorRecord)
		}

		foreach($currentResolvedPath in $currentResolvedPaths)
		{
			$resolvedPaths += $currentResolvedPath.ProviderPath
		}
	}

	$resolvedPaths
}

function IsValidFileSystemPath 
{
	param 
	(
		[string[]] $path
	)
	
	$result = $true;

	# null and empty check are already done on Path parameter at the cmdlet layer.
	foreach($currentPath in $path)
	{
		if(!([System.IO.File]::Exists($currentPath) -or [System.IO.Directory]::Exists($currentPath)))
		{
			$errorMessage = ("The path '{0}' either does not exist or is not a valid file system path." -f $currentPath)
			ThrowTerminatingErrorHelper 'PathNotFound' $errorMessage ([System.Management.Automation.ErrorCategory]::InvalidArgument) $currentPath
		}
	}

	return $result;
}

function ValidateDuplicateFileSystemPath 
{
	param 
	(
		[string] $inputParameter,
		[string[]] $path
	)
	
	$uniqueInputPaths = @()

	# null and empty check are already done on Path parameter at the cmdlet layer.
	foreach($currentPath in $path)
	{
		$currentInputPath = $currentPath.ToUpper()
		if($uniqueInputPaths.Contains($currentInputPath))
		{
			$errorMessage = ("The input to {0} parameter contains a duplicate path '{1}'. Provide a unique set of paths as input to {2} parameter." -f $inputParameter, $currentPath, $inputParameter)
			ThrowTerminatingErrorHelper 'DuplicatePathFound' $errorMessage ([System.Management.Automation.ErrorCategory]::InvalidArgument) $currentPath
		}
		else
		{
			$uniqueInputPaths += $currentInputPath
		}
	}
}

function CompressArchiveHelper 
{
	param 
	(
		[string[]] $sourcePath,
		[string]   $rootPath,
		[string]   $destinationPath
	)

	$numberOfItemsArchived = 0
	$sourceFilePaths = @()
	$sourceDirPaths = @()

	foreach($currentPath in $sourcePath)
	{
		$result = Test-Path $currentPath -Type Leaf
		if($result -eq $true)
		{
			$sourceFilePaths += $currentPath
		}
		else
		{
			$sourceDirPaths += $currentPath
		}
	}

	# The Source Path contains one or more directory (this directory can have files under it) and no files to be compressed.
	if($sourceFilePaths.Count -eq 0 -and $sourceDirPaths.Count -gt 0)
	{
		$currentSegmentWeight = 100/[double]$sourceDirPaths.Count
		$previousSegmentWeight = 0
		foreach($currentSourceDirPath in $sourceDirPaths)
		{
			$count = CompressSingleDirHelper $currentSourceDirPath $rootPath $destinationPath $true $previousSegmentWeight $currentSegmentWeight
			$numberOfItemsArchived += $count
			$previousSegmentWeight += $currentSegmentWeight
		}
	}

	# The Source Path contains only files to be compressed.
	elseif($sourceFilePaths.Count -gt 0 -and $sourceDirPaths.Count -eq 0)
	{
		# $previousSegmentWeight is equal to 0 as there are no prior segments.
		# $currentSegmentWeight is set to 100 as all files have equal weightage.
		$previousSegmentWeight = 0
		$currentSegmentWeight = 100

		$numberOfItemsArchived = CompressFilesHelper $sourceFilePaths $rootPath $destinationPath $previousSegmentWeight $currentSegmentWeight
	}
	# The Source Path contains one or more files and one or more directories (this directory can have files under it) to be compressed.
	elseif($sourceFilePaths.Count -gt 0 -and $sourceDirPaths.Count -gt 0)
	{
		# each directory is considered as an individual segments & all the individual files are clubed in to a separate segment.
		$currentSegmentWeight = 100/[double]($sourceDirPaths.Count +1)
		$previousSegmentWeight = 0

		foreach($currentSourceDirPath in $sourceDirPaths)
		{
			$count = CompressSingleDirHelper $currentSourceDirPath $rootPath $destinationPath $previousSegmentWeight $currentSegmentWeight
			$numberOfItemsArchived += $count
			$previousSegmentWeight += $currentSegmentWeight
		}

		$count = CompressFilesHelper $sourceFilePaths $rootPath $destinationPath $previousSegmentWeight $currentSegmentWeight
		$numberOfItemsArchived += $count
	}

	return $numberOfItemsArchived
}

function CompressFilesHelper 
{
	param 
	(
		[string[]] $sourceFilePaths,
		[string]   $rootPath,
		[string]   $destinationPath,
		[double]   $previousSegmentWeight,
		[double]   $currentSegmentWeight
	)
			
	$numberOfItemsArchived = ZipArchiveHelper $sourceFilePaths $destinationPath $rootPath $previousSegmentWeight $currentSegmentWeight

	return $numberOfItemsArchived
}

function CompressSingleDirHelper 
{
	param 
	(
		[string] $sourceDirPath,
		[string] $rootPath,
		[string] $destinationPath,
		[double] $previousSegmentWeight,
		[double] $currentSegmentWeight
	)

	$subDirFiles = @()
	$dirContents = Get-ChildItem $sourceDirPath -Recurse
	foreach($currentContent in $dirContents)
	{
		$result = Test-Path $currentContent.FullName -Type Leaf
		if($result -eq $true)
		{
			$subDirFiles += $currentContent.FullName
		}
		else
		{
			# The currentContent points to a directory.
			# We need to check if the directory is an empty directory, if so such a
			# directory has to be explicitly added to the archive file.
			# if there are no files in the directory the GetFiles() API returns an empty array.
			$files = $currentContent.GetFiles()
			if($files.Count -eq 0)
			{
				$subDirFiles += $currentContent.FullName + '\'
			}
		}
	}

	$sourcePaths = $subDirFiles

	$numberOfItemsArchived = ZipArchiveHelper $sourcePaths $destinationPath $rootPath $previousSegmentWeight $currentSegmentWeight

	return $numberOfItemsArchived
}

function ZipArchiveHelper 
{
	param 
	(
		[string[]] $sourcePaths,
		[string]   $destinationPath,
		[string]   $modifiedSourceDirFullName,
		[double]   $previousSegmentWeight,
		[double]   $currentSegmentWeight
	)

	$numberOfItemsArchived = 0
	$fileMode = [System.IO.FileMode]::Create
	$result = Test-Path $DestinationPath -Type Leaf
	if($result -eq $true)
	{
		$fileMode = [System.IO.FileMode]::Open
	}

	Add-Type -AssemblyName System.IO.Compression

	try
	{
		# At this point we are sure that the archive file has write access.
		$archiveFileStreamArgs = @($destinationPath, $fileMode)
		$archiveFileStream = New-Object -TypeName System.IO.FileStream -ArgumentList $archiveFileStreamArgs

		$zipArchiveArgs = @($archiveFileStream, [System.IO.Compression.ZipArchiveMode]::Update, $false)
		$zipArchive = New-Object -TypeName System.IO.Compression.ZipArchive -ArgumentList $zipArchiveArgs

		$currentEntryCount = 0
		$progressBarStatus = ("The archive file '{0}' creation is in progress..." -f $destinationPath)
		$bufferSize = 4kb
		$buffer = New-Object byte[] $bufferSize

		foreach($currentFilePath in $sourcePaths)
		{
			if($modifiedSourceDirFullName -ne $null -and $modifiedSourceDirFullName.Length -gt 0)
			{
				$index = $currentFilePath.IndexOf($modifiedSourceDirFullName, [System.StringComparison]::OrdinalIgnoreCase)
				$currentFilePathSubString = $currentFilePath.Substring($index, $modifiedSourceDirFullName.Length)
				$relativeFilePath = $currentFilePath.Replace($currentFilePathSubString, '').Trim()
			}
			else
			{
				$relativeFilePath = [System.IO.Path]::GetFileName($currentFilePath)
			}

			# If a directory needs to be added to an archive file, 
			# by convention the .Net API's expect the path of the directory 
			# to end with '\' to detect the path as an directory.
			if(!$relativeFilePath.EndsWith('\', [StringComparison]::OrdinalIgnoreCase))
			{
				try
				{
					try
					{
						$currentFileStream = [System.IO.File]::Open($currentFilePath, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read)
					}
					catch
					{
						# Failed to access the file. Write a non terminating error to the pipeline 
						# and move on with the remaining files.
						$exception = $_.Exception
						if($null -ne $_.Exception -and 
						$null -ne $_.Exception.InnerException)
						{
							$exception = $_.Exception.InnerException
						}
						$errorRecord = CreateErrorRecordHelper 'CompressArchiveUnauthorizedAccessError' $null ([System.Management.Automation.ErrorCategory]::PermissionDenied) $exception $currentFilePath
						Write-Error -ErrorRecord $errorRecord
					}

					if($null -ne $currentFileStream)
					{
						$srcStream = New-Object System.IO.BinaryReader $currentFileStream

						$currentArchiveEntry = $zipArchive.CreateEntry("$relativeFilePath", [System.IO.Compression.CompressionLevel]::Optimal)
						$destStream = New-Object System.IO.BinaryWriter $currentArchiveEntry.Open()
					
						while($numberOfBytesRead = $srcStream.Read($buffer, 0, $bufferSize))
						{
							$destStream.Write($buffer, 0, $numberOfBytesRead)
							$destStream.Flush()
						}                    

						$numberOfItemsArchived += 1
						$addItemtoArchiveFileMessage = ("Adding '{0}'." -f $currentFilePath)
					}
				}
				catch {
					Write-Output $_
				}
				finally
				{
					if($null -ne $currentFileStream)
					{
						$currentFileStream.Dispose()
					}
					if($null -ne $srcStream)
					{
						$srcStream.Dispose()
					}
					if($null -ne $destStream)
					{
						$destStream.Dispose()
					}
				}
			}
			else
			{
				$relativeFilePath = $relativeFilePath -replace '\\', '/'

				$currentArchiveEntry = $zipArchive.CreateEntry("$relativeFilePath", [System.IO.Compression.CompressionLevel]::Optimal)
				$numberOfItemsArchived += 1
				$addItemtoArchiveFileMessage = ("Adding '{0}'." -f $currentFilePath)
			}

			if($null -ne $addItemtoArchiveFileMessage)
			{
				Write-Verbose $addItemtoArchiveFileMessage
			}

			$currentEntryCount += 1
			ProgressBarHelper 'Compress-Archive' $progressBarStatus $previousSegmentWeight $currentSegmentWeight $sourcePaths.Count  $currentEntryCount
		}
	}
	finally
	{
		if($null -ne $zipArchive)
		{
			$zipArchive.Dispose()
		}

		if($null -ne $archiveFileStream)
		{
			$archiveFileStream.Dispose()
		}

		# Complete writing progress.
		Write-Progress -Activity 'Compress-Archive' -Completed
	}

	return $numberOfItemsArchived
}

<############################################################################################ 
		# ProgressBarHelper: This is a helper function used to display progress message.
		# This function is used by both Compress-Archive & Expand-Archive to display archive file
		# creation/expansion progress.
############################################################################################>
function ProgressBarHelper 
{
	param 
	(
		[string] $cmdletName,
		[string] $status,
		[double] $previousSegmentWeight,
		[double] $currentSegmentWeight,
		[int]    $totalNumberofEntries,
		[int]    $currentEntryCount
	)

	if($currentEntryCount -gt 0 -and 
		$totalNumberofEntries -gt 0 -and 
		$previousSegmentWeight -ge 0 -and 
	   $currentSegmentWeight -gt 0)
	{
		$entryDefaultWeight = $currentSegmentWeight/[double]$totalNumberofEntries

		$percentComplete = $previousSegmentWeight + ($entryDefaultWeight * $currentEntryCount)
		Write-Progress -Activity $cmdletName -Status $status -PercentComplete $percentComplete 
	}
}

<############################################################################################ 
		# CSVHelper: This is a helper function used to append comma after each path specified by
		# the SourcePath array. This helper function is used to display all the user supplied paths
		# in the WhatIf message.
############################################################################################>
function CSVHelper 
{
	param 
	(
		[string[]] $sourcePath
	)

	# SourcePath has already been validated by the calling function.
	if($sourcePath.Count -gt 1)
	{
		$sourcePathInCsvFormat = "`n"
		for($currentIndex=0; $currentIndex -lt $sourcePath.Count; $currentIndex++)
		{
			if($currentIndex -eq $sourcePath.Count - 1)
			{
				$sourcePathInCsvFormat += $sourcePath[$currentIndex]
			}
			else
			{
				$sourcePathInCsvFormat += $sourcePath[$currentIndex] + "`n"
			}
		}
	}
	else
	{
		$sourcePathInCsvFormat = $sourcePath
	}

	return $sourcePathInCsvFormat
}

<############################################################################################ 
		# ThrowTerminatingErrorHelper: This is a helper function used to throw terminating error.
############################################################################################>
function ThrowTerminatingErrorHelper 
{
	param 
	(
		[string] $errorId,
		[string] $errorMessage,
		[System.Management.Automation.ErrorCategory] $errorCategory,
		[object] $targetObject,
		[Exception] $innerException
	)

	if($innerException -eq $null)
	{
		$exception = New-Object System.IO.IOException $errorMessage
	}
	else
	{
		$exception = New-Object System.IO.IOException $errorMessage, $innerException
	}

	$exception = New-Object System.IO.IOException $errorMessage
	$errorRecord = New-Object System.Management.Automation.ErrorRecord $exception, $errorId, $errorCategory, $targetObject
	$PSCmdlet.ThrowTerminatingError($errorRecord)
}

<############################################################################################ 
		# CreateErrorRecordHelper: This is a helper function used to create an ErrorRecord
############################################################################################>
function CreateErrorRecordHelper 
{
	param 
	(
		[string] $errorId,
		[string] $errorMessage,
		[System.Management.Automation.ErrorCategory] $errorCategory,
		[Exception] $exception,
		[object] $targetObject
	)

	if($null -eq $exception)
	{
		$exception = New-Object System.IO.IOException $errorMessage
	}

	$errorRecord = New-Object System.Management.Automation.ErrorRecord $exception, $errorId, $errorCategory, $targetObject
	return $errorRecord
}

Export-ModuleMember -function *-*