package com.xebialabs.deployit.ascode.service.spec.util

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.deployit.ascode.yaml.model.DeploymentSpec
import com.xebialabs.deployit.core.util.IdExtensions._
import com.xebialabs.deployit.engine.api.dto.Deployment
import com.xebialabs.deployit.plugin.api.udm.{DeployedApplication}
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.server.api.util.IdGenerator
import com.xebialabs.deployit.service.deployment.DeploymentTaskService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service



@Service
@Autowired
class DeploymentPreparationService(deploymentTaskService: DeploymentTaskService,
                                   repositoryService: RepositoryService) {
  private def prepareDeployment(version: String, environment: String): Deployment = {
    val applicationId = version.getParent
    if (deploymentTaskService.isDeployed(applicationId, environment)) {
      deploymentTaskService.prepareUpdate(version, IdGenerator.generateId(environment, applicationId.getName))
    } else {
      deploymentTaskService.prepareInitial(version, environment)
    }
  }

  def prepareUnDeployment(spec: DeploymentSpec): Deployment = {
    val deployment = deploymentTaskService.prepareUndeploy(spec.version)
    if(deployment == null)
      throw AsCodeException.throwDocumentFieldException("Application Version Id",
      s"Application version '${spec.version}' is not deployed in '${spec.environment}'")
    deployment
  }


  private def fillInDeployedApplicationFields(deployment: Deployment, spec: DeploymentSpec): Unit = {
    val deployedApplication = deployment.getDeployedApplication.asInstanceOf[DeployedApplication]
    spec.deployedApplication.setId(deployedApplication.getId)
    spec.deployedApplication.setVersion(deployedApplication.getVersion)
    spec.deployedApplication.setEnvironment(deployedApplication.getEnvironment)
    deployment.setDeployedApplication(spec.deployedApplication)
  }

  private def readDeployedApplication(unDeployment: Deployment, spec: DeploymentSpec): Unit = {
    val undeployDeployedApplication = unDeployment.getDeployedApplication().asInstanceOf[DeployedApplication]
    val deployedApplication = repositoryService.read(undeployDeployedApplication.getId,
      false).asInstanceOf[DeployedApplication]
    undeployDeployedApplication.setDeployeds(deployedApplication.getDeployeds)
    undeployDeployedApplication.setUndeployDependencies(spec.deployedApplication.isUndeployDependencies)
  }

  def deploymentSpecToDeployment(spec: DeploymentSpec): Deployment = {
    val deployment = prepareDeployment(spec.version, spec.environment)
    val deploymentWithDeployeds = deploymentTaskService.prepareAutoDeployeds(deployment)
    fillInDeployedApplicationFields(deploymentWithDeployeds, spec)
    deploymentWithDeployeds
  }

  def deploymentSpecToUnDeployment(spec: DeploymentSpec): Deployment = {
    val unDeployment = prepareUnDeployment(spec)
    readDeployedApplication(unDeployment, spec)
    unDeployment
  }
}
