package com.xebialabs.deployit.ascode.service.spec.permission

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.permission.PermissionHandler.Level._
import com.xebialabs.deployit.core.util.IdExtensions._

object PermissionIndex {
  def build(permissions: List[Permission]): PermissionIndex = permissions
    .foldLeft(PermissionIndex()) { case (acc, permission) =>
      val level = permission.getLevel
      val root = permission.getRoot
      val isCiPermission = level == CI || level == BOTH

      val global = if (level == GLOBAL || level == BOTH) acc.global + permission.getPermissionName else acc.global
      val ciPermissions = if (isCiPermission && root == null) acc.ciPermissions + permission.getPermissionName else acc.ciPermissions
      val rootPermissions = if (isCiPermission && root != null) {
        val rootPerms = acc.rootPermissions.getOrElse(root.getRootNodeName, Set[String]()) + permission.getPermissionName
        acc.rootPermissions + (root.getRootNodeName -> rootPerms)
      } else acc.rootPermissions
      PermissionIndex(global, ciPermissions, rootPermissions)
    }
}

case class PermissionIndex(global: Set[String] = Set(),
                           ciPermissions: Set[String] = Set(),
                           rootPermissions: Map[String, Set[String]] = Map()) {
  private def checkGlobalPermission(permission: String): Unit = {
    if (!global.contains(permission)) {
      throw new AsCodeException(s"Invalid global permission [$permission]")
    }
  }

  private def checkCiPermission(ciId: String, permission: String): Unit = {
    val root = ciId.getRoot
    val allCis = ciPermissions ++ rootPermissions.getOrElse(root, Set())
    if (!allCis.contains(permission)) {
      throw new AsCodeException(s"Invalid permission [$permission] for ci [$ciId]")
    }
  }

  def checkPermission(ciId: Option[String]): String => Any = permission => ciId match {
    case Some(ci) => checkCiPermission(ci, permission)
    case None => checkGlobalPermission(permission)
  }
}