package com.xebialabs.deployit.ascode.service.spec

import com.xebialabs.ascode.service.spec.{InterpreterContext, SpecInterpreter}
import com.xebialabs.ascode.yaml.dto.AsCodeResponse
import com.xebialabs.ascode.yaml.model.CiSpec
import com.xebialabs.deployit.ascode.service.validation.PermissionValidator
import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider
import com.xebialabs.deployit.engine.spi.command.util.Update
import com.xebialabs.deployit.io.ArtifactFileUtils
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_REPO
import com.xebialabs.deployit.service.validation.Validator
import com.xebialabs.xldeploy.packager.placeholders.SourceArtifactScanner
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import java.util.Optional

import com.xebialabs.deployit.core.sql.RepositoryHelper

import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.language.implicitConversions

@Component
@Autowired
class CiSpecInterpreter(repositoryHelper: RepositoryHelper,
                        permissionValidator: PermissionValidator,
                        scanner: SourceArtifactScanner,
                        validator: Validator,
                        checksumAlgorithmProvider: ChecksumAlgorithmProvider) extends SpecInterpreter {
  override def isDefinedAt(context: InterpreterContext): Boolean = context.definition.spec.isInstanceOf[CiSpec]

  private implicit def updatesToCiList(updates: mutable.Buffer[Update]): List[String] =
    updates.map(_.getNewCi.getId).toList

  private def verify(cis: List[ConfigurationItem]): Unit = {
    validator.validateCis(cis.asJava)
    permissionValidator.checkPermissions(EDIT_REPO, cis.map(_.getId))
  }

  private def handleUriArtifacts(cis: List[ConfigurationItem]): Unit = {
    cis.flatMap {
      case artifact: SourceArtifact if artifact.getFileUri != null => Some(artifact)
      case _ => None
    }.foreach(ArtifactFileUtils.handleArtifact(scanner, _, checksumAlgorithmProvider))
  }

  private def createOrUpdateCis(cis: List[ConfigurationItem], scmData: Option[SCMTraceabilityData]) = {
    handleUriArtifacts(cis)
    val (updated, created) = repositoryHelper
      .createOrUpdateCis(cis.asJava, Optional.empty(), scmData.orNull)
      .asScala
      .partition(_.getPreviousCi != null)

    AsCodeResponse.cis(created, updated)
  }

  override def apply(context: InterpreterContext): AsCodeResponse = {
    val cis = context.definition.spec.asInstanceOf[CiSpec].cis
    verify(cis)
    createOrUpdateCis(cis, context.scmTraceabilityData)
  }
}
