package com.xebialabs.deployit.ascode.yaml.parser.spec.deployment

import com.fasterxml.jackson.databind.{JsonNode, ObjectMapper}
import com.xebialabs.ascode.schema.JsonSchema._
import com.xebialabs.ascode.utils.TypeSugar._
import com.xebialabs.ascode.yaml.model.Spec
import com.xebialabs.ascode.yaml.parser.SpecParser
import com.xebialabs.ascode.yaml.parser.support.{JacksonCiReader, ReaderConfig}
import com.xebialabs.ascode.yaml.parser.util.JsonNodeSugar._
import com.xebialabs.deployit.ascode.schema.JsonSchemaGenerator._
import com.xebialabs.deployit.ascode.yaml.model.Constants.Kinds.STITCH_DRYRUN
import com.xebialabs.deployit.ascode.yaml.model.Constants.Versions._
import com.xebialabs.deployit.ascode.yaml.model.{Constants, DeploymentSpec}
import com.xebialabs.deployit.ascode.yaml.parser.util.RelativeIdResolver
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import com.xebialabs.xltype.serialization.ConfigurationItemConverter

import scala.collection.immutable.ListMap
import scala.jdk.CollectionConverters._

object DeploymentSpecParser extends SpecParser with SchemaGenerator {
  private val definitionKey = "deploymentKindSpec"
  private val reservedProperties = Set("package", "environment", "id", "type", "stitchSource", "branchName")
  private val overrides = Map("orchestrators" -> "orchestrator", "package" -> "version")

  override def parse(definitionKind: String,
                     metadata: Option[Map[String, String]],
                     node: JsonNode)(implicit mapper: ObjectMapper): Spec = {
    val config = ReaderConfig(
      overrides = overrides,
      reserved = reservedProperties,
      idResolver = new RelativeIdResolver(definitionKind, metadata)
    )
    val reader = new JacksonCiReader(node, None, config) {
      override def getType: String = typeOf[DeployedApplication].toString
    }
    val converter = new ConfigurationItemConverter()
    val version = node.requiredString("package")
    val environment = node.requiredString("environment")
    val deploymentSpec = converter.readCi(reader).asInstanceOf[DeployedApplication]

    if (definitionKind == STITCH_DRYRUN) {
      val stitchSource = node.requiredString("stitchSource")
      val branchName = node.requiredString("branchName")
      DeploymentSpec(version, environment, deploymentSpec, converter.getReferences.asScala.toList, stitchSource, branchName)
    } else {
      DeploymentSpec(version, environment, deploymentSpec, converter.getReferences.asScala.toList)
    }
  }

  override def generateSchema(definitionKind: String)(implicit generatorContext: SchemaGeneratorContext): SchemaNode = {
    node
      .additionalProperties(false)
      .required("apiVersion", "kind", "spec")
      .properties(
        version(XLD_API_VERSION),
        kind(definitionKind),
        metadata(node.properties(imports())),
        spec(refNode(s"${buildPrefix(Constants.Schema.PRODUCT_PREFIX)}.$definitionKey").`type`("object"))
      )
  }

  override def generateDefinitions(implicit generatorContext: SchemaGeneratorContext): Map[String, ListMap[String, Any]] = {
    val definition =
      getTypeDefinition(typeOf[DeployedApplication].getDescriptor)
        .filterOverrideProperties(Set("id", "type", "name", "deployeds"), overrides.map { case (key, value) => value -> key })
        .addRequired("package", "environment")
    Map(s"$xlNamespace.$definitionKey" -> definition)
  }
}
