package com.xebialabs.ascode.yaml.writer.support

import java.util

import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator
import com.xebialabs.ascode.utils.TypeSugar._
import com.xebialabs.ascode.utils.Utils
import com.xebialabs.ascode.yaml.sugar.{SugarConfig, Sugarizer}
import com.xebialabs.deployit.core.rest.YamlSupport._
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.{CiAttributes, ExternalProperty}
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage
import com.xebialabs.xltype.serialization.CiWriter

import scala.jdk.CollectionConverters._

case class TaggedValue(tag: String, value: Any)

class JacksonCiWriter(yamlGenerator: YAMLGenerator)(implicit val sugarConfig: SugarConfig) extends CiWriter {
  override def externalProperties(externalProperties: util.Map[String, ExternalProperty]): Unit = {}

  private def writeWithSugar(field: String, value: String)(implicit ciType: Type): Unit = {
    Sugarizer.writeWithSugar(field, value).foreach { case (k, v) => yamlGenerator.writeStringField(k, v) }
  }

  private var relativePath: String = _

  override def startList(): Unit = yamlGenerator.writeStartArray()

  override def endList(): Unit = yamlGenerator.writeEndArray()

  override def startCi(`type`: String, id: String): Unit = {
    if (relativePath == null) {
      relativePath = getParent(id)
    }
    yamlGenerator.writeStartObject()

    implicit val ciType: Type = typeOf(`type`)
    if (id != null)
      writeWithSugar("name", getCorrectName(id))
    writeWithSugar("type", `type`)
  }

  override def endCi(): Unit = yamlGenerator.writeEndObject()

  override def token(token: String): Unit = {}

  override def ciAttributes(ciAttributes: CiAttributes): Unit = {}

  override def ciFileAttribute(file: String): Unit = {}

  override def startProperty(name: String): Unit = yamlGenerator.writeFieldName(name)

  override def endProperty(): Unit = {}

  override def valueAsString(rawValue: scala.Any): Unit = {
    Option(rawValue) match {
      case Some(TaggedValue(tag, value)) => yamlGenerator.writeTaggedString(tag, value.toString)
      case Some(double: Double) => yamlGenerator.writeNumber(double)
      case Some(int: Int) => yamlGenerator.writeNumber(int)
      case Some(bool: Boolean) => yamlGenerator.writeBoolean(bool)
      case Some(value) => yamlGenerator.writeString(value.toString)
      case None => yamlGenerator.writeNull()
    }
  }

  override def valuesAsStrings(values: util.Collection[_]): Unit = {
    yamlGenerator.writeStartArray()
    values.asScala.foreach(valueAsString)
    yamlGenerator.writeEndArray()
  }

  override def mapAsStrings(map: util.Map[_, _]): Unit = {
    yamlGenerator.writeStartObject()
    map.asScala.foreach { case (k, v) =>
      Option(v) match {
        case Some(TaggedValue(tag, value)) => yamlGenerator.writeTaggedStringField(k.toString, tag, value.toString)
        case Some(value) => yamlGenerator.writeStringField(k.toString, value.toString)
        case None => yamlGenerator.writeNull()
      }
    }
    yamlGenerator.writeEndObject()
  }

  override def ciReference(reference: String): Unit = valueAsString(reference)

  override def ciReferences(references: util.Collection[String]): Unit = valuesAsStrings(references)

  override def typedCiReference(ci: ConfigurationItemId): Unit = {}

  override def typedCiReferences(references: util.Collection[_ <: ConfigurationItemId]): Unit = {}

  override def validationMessages(validations: util.List[ValidationMessage]): Unit = {}

  private def getParent(id: String): String = {
    Option(id).getOrElse("").lastIndexOf('/') match {
      case lastSlashPos if lastSlashPos > 0 => id.substring(0, lastSlashPos)
      case _ => ""
    }
  }

  private def getCorrectName(id: String): String = {
    if (getParent(id).equals(relativePath)) id else Utils.getName(id).orNull
  }
}
