package com.xebialabs.ascode.schema

import com.xebialabs.ascode.utils.TypeSugar._
import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, DescriptorRegistry, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.{BaseDeployedContainer, Deployed}

import scala.jdk.CollectionConverters._
import scala.collection.immutable.ListMap

object JsonSchema {

  trait SchemaGenerator {
    def generateSchema(definitionKind: String)(implicit generatorContext: SchemaGeneratorContext): SchemaNode
    def generateDefinitions(implicit generatorContext: SchemaGeneratorContext): Map[String, ListMap[String, Any]] = Map.empty
  }

  object SchemaGeneratorContext {
    def apply(descriptors: List[Descriptor]): SchemaGeneratorContext = {
      val typeContainmentMap: Map[Type, Set[Type]] = descriptors
        .to(LazyList)
        .filter(d => !d.isVirtual && (d.isInstance[BaseDeployedContainer[_, _]] || !d.isInstance[Deployed[_, _]]))
        .flatMap(d => d.getPropertyDescriptors.asScala.to(LazyList).map(d -> _))
        .filter { case (_, pd) => pd.isAsContainment && pd.getKind == PropertyKind.CI }
        .flatMap { case (d, pd) =>
          val subtypes = DescriptorRegistry
            .getSubtypes(pd.getReferencedType)
            .asScala
            .to(LazyList)
            .map(_.getDescriptor)
            .filterNot(_.isVirtual)
            .map(st => st.getType -> d.getType)
          (pd.getReferencedType -> d.getType) #:: subtypes
        }
        .groupBy { case (key, _) => key }
        .view
        .mapValues(_.map { case (_, value) => value }.toSet)
        .toMap
      SchemaGeneratorContext(descriptors, typeContainmentMap)
    }
  }
  case class SchemaGeneratorContext(allTypes: List[Descriptor] = Nil, typeContainmentMap: Map[Type, Set[Type]] = Map.empty)

  val xlNamespace = "com.xl"
  val definitionsPrefix = "#/definitions/"

  type SchemaNode = ListMap[String, Any]

  def node = new SchemaNode

  def refNode(id: String): SchemaNode = ListMap("$ref" -> id)

  def definitions(defs: (String, SchemaNode)*): ListMap[String, ListMap[String, Any]] = ListMap(
    defs.map { case (key, node) => key -> node }: _*
  )

  def versionNode(value: String): SchemaNode = node
    .`type`("string")
    .description("The version of the API")
    .enum(value)

  def version(value: String): (String, SchemaNode) = "apiVersion" -> versionNode(value)

  def kindNode(value: String): SchemaNode = node
    .`type`("string")
    .description("The kind is an identifier for the type of YAML file.")
    .enum(value)

  def kind(value: String): (String, SchemaNode) = "kind" -> kindNode(value)

  def metadataNode(metadataPart: SchemaNode): SchemaNode = node
    .`type`("object")
    .description("Metadata")
    .mergePart(metadataPart)

  def metadata(metadataPart: SchemaNode = ListMap.empty): (String, SchemaNode) = "metadata" -> metadataNode(metadataPart)

  def specNode(specPart: SchemaNode): SchemaNode = node
    .`type`("array")
    .description("The spec object contains the specification of the kind.")
    .mergePart(specPart)

  def spec(specPart: SchemaNode): (String, SchemaNode) = "spec" -> specNode(specPart)

  def importsNode(): SchemaNode = node
    .`type`("array")
    .description("Path to imports")
    .minItems(1)
    .uniqueItems(true)
    .items(node.`type`("string"))

  def imports(): (String, SchemaNode) = "imports" -> importsNode()

  implicit class SchemaMapSugar(map: SchemaNode) {
    def id(id: String): SchemaNode = map + ("$id" -> id)

    def schema(schema: String): SchemaNode = map + ("$schema" -> schema)

    def `type`(`type`: String): SchemaNode = map + ("type" -> `type`)

    def description(description: String): SchemaNode = map + ("description" -> description)

    def enum(value: String*): SchemaNode = map + ("enum" -> value)

    def additionalProperties(props: Boolean): SchemaNode = map + ("additionalProperties" -> props)

    def required(required: String*): SchemaNode = map + ("required" -> required.toList)

    def minItems(minItems: Long): SchemaNode = map + ("minItems" -> minItems)

    def uniqueItems(uniqueItems: Boolean): SchemaNode = map + ("uniqueItems" -> uniqueItems)

    def items(items: SchemaNode): SchemaNode = map + ("items" -> items)

    def properties(properties: SchemaNode): SchemaNode = map + ("properties" -> properties)

    def properties(props: (String, SchemaNode)*): SchemaNode = properties(ListMap(props: _*))

    def oneOf(oneOf: List[Map[String, Any]]): SchemaNode = map + ("oneOf" -> oneOf)

    def definitions(definitions: Map[String, ListMap[String, Any]]): SchemaNode = map + ("definitions" -> definitions)

    def default(defaultValue: Any): SchemaNode = map + ("default" -> defaultValue)

    def patternProperties(props: (String, SchemaNode)*): SchemaNode = map + ("patternProperties" -> ListMap(props: _*))

    def mergePart(another: SchemaNode): SchemaNode = map ++ another

    def transformProperties(op: SchemaNode => SchemaNode): SchemaNode = {
      val props = map
        .getOrElse("properties", ListMap.empty[String, Any])
        .asInstanceOf[SchemaNode]
      map.properties(op(props))
    }

    def transformRequired(op: List[String] => List[String]): SchemaNode = {
      val req = map
        .getOrElse("required", Nil)
        .asInstanceOf[List[String]]
      map.required(op(req): _*)
    }

    def filterOverrideProperties(forbiddenProps: Set[String], overrides: Map[String, String]): SchemaNode = {
      def filterOverrideKey(key: String) =
        Some(key)
          .filterNot(forbiddenProps.contains)
          .map(overrides.getOrElse(_, key))

      map
        .transformProperties { props => props.flatMap { case (key, value) => filterOverrideKey(key).map(_ -> value) } }
        .transformRequired { req => req.flatMap(filterOverrideKey) }
    }

    def addRequired(newValues: String*): SchemaNode = transformRequired(reqs => (reqs ::: newValues.toList).distinct)
  }

  def buildPrefix(productPrefix: String): String = {
    s"$definitionsPrefix$productPrefix$xlNamespace"
  }

  def buildRefFromType(`type`: Type, productPrefix: String): SchemaNode = refNode(s"${buildPrefix(productPrefix)}.${`type`.toString}")

  def buildRefList(types: List[Type], productPrefix: String): List[SchemaNode] = types.map(buildRefFromType(_, productPrefix))

  def buildRefListForRoot(root: String, productPrefix: String)(implicit generatorContext: SchemaGeneratorContext): List[SchemaNode] = buildRefList(
    generatorContext.allTypes
      .withFilter(_.getRoot.getRootNodeName == root)
      .map(_.getType),
    productPrefix
  )

  def buildSchemaMerge(product: String, node: SchemaNode): SchemaNode = {
    val schema = new SchemaMapSugar(new SchemaNode)

    schema
      .schema("https://json-schema.org/draft-07/schema#")
      .description("A description of objects in the XL Products")
      .`type`("object")
      .oneOf(
        List(Map("$ref" -> s"$definitionsPrefix$product"))
      )
      .definitions(Map(product -> node))
  }
}
