package com.xebialabs.deployit.upgrade

import com.xebialabs.deployit.server.api.upgrade.Version.valueOf
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, UpgradeException, Version}
import com.xebialabs.xlplatform.upgrade.RepositoryVersionService
import org.apache.commons.lang.mutable.MutableBoolean
import org.junit.Assert.assertTrue
import org.scalatest.funspec.AnyFunSpecLike
import org.scalatest.matchers.should.Matchers
import org.springframework.context.support.StaticApplicationContext

import java._
import scala.collection.mutable.ArrayBuffer

trait RepositoryUpgraderTest extends AnyFunSpecLike with Matchers {

  implicit val versionService: RepositoryVersionService
  implicit val upgradeStrategy: UpgradeStrategy

  private var a_1: Upgrade = _
  private var a_2: Upgrade = _
  private var a_2_second_upgrade: Upgrade = _
  private var a_3: Upgrade = _
  private var c_1: Upgrade = _
  private var b_2: Upgrade = _

  private var upgradesApplied: ArrayBuffer[Upgrade] = _
  private var upgrader: Upgrader = _

  protected def initRepo(): Unit

  def setup(): Unit = {
    initRepo()

    versionService.readVersionOfComponent("b") shouldBe null
    upgradesApplied = ArrayBuffer.empty

    a_1 = new TestUpgrade(valueOf("a", "1.0.0"))
    b_2 = new TestUpgrade(valueOf("b", "2.0.0"))
    a_2 = new TestUpgrade(valueOf("a", "2.0.0"))
    a_2_second_upgrade = new TestUpgrade(valueOf("a", "2.0.0"))
    a_3 = new TestUpgrade(valueOf("a", "2.0.1"))
    c_1 = new TestUpgrade(valueOf("c", "1.0.0"))

    upgrader = new TestUpgrader(upgradeStrategy, true)

    addDefaultUpgrades(upgrader)
  }

  describe("Repository upgrader") {

    it("should apply only needed upgrades") {
      versionService.storeVersionOfComponent(valueOf("a", "1.0.1"))
      upgrader.applyUpgrades()
      upgradesApplied should contain.allOf(a_2, a_3)
      upgradesApplied should not contain(a_1, b_2, c_1)
      versionService.readVersionOfComponent("a").getVersion shouldBe "2.0.1"
    }

    it("should apply all upgrades of the same version") {
      versionService.storeVersionOfComponent(valueOf("a", "1.0.1"))
      upgrader.applyUpgrades()
      upgradesApplied should contain.allOf(a_2, a_2_second_upgrade, a_3)
      versionService.readVersionOfComponent("a").getVersion shouldBe "2.0.1"
    }

    it("should apply no ugrades when no version previously stored") {
      versionService.storeVersionOfComponent(valueOf("b", "0.0.0"))
      upgrader.applyUpgrades()
      upgradesApplied.length shouldBe 0
      versionService.readVersionOfComponent("b").getVersion shouldBe "2.0.0"
    }

    it("should skip upgrades if plugin version not found") {
      versionService.readVersionOfComponent("b") shouldBe null
      upgrader.upgradeComponent("b")
      upgradesApplied.length shouldBe 0
      versionService.readVersionOfComponent("b") shouldBe null
    }

    it("should not apply upgrades on an unknown component") {
      upgrader.addComponent("unknown_component")
      upgrader.applyUpgrades()
      upgradesApplied.length shouldBe 0
    }

    it("should not apply upgrades of known component without upgrades") {
      upgrader.addComponent("other_component")
      versionService.storeVersionOfComponent(valueOf("other_component", "0.0.1"))
      upgrader.applyUpgrades()
      upgradesApplied.length shouldBe 0
    }

    it("should upgrade deployit component first") {
      val foundComponents = new ArrayBuffer[String]
      val testUpgrader = new TestUpgrader(upgradeStrategy, true) {
        override private[upgrade] def upgradeComponent(component: String): Unit = foundComponents += component
        override def findUpgrades = new util.HashMap[String, util.List[Upgrade]]
      }
      testUpgrader.autoUpgrade()
      foundComponents.head shouldBe "deployit"
    }

    it("should ask question when force upgrades is set to false") {
      versionService.storeVersionOfComponent(valueOf("a", "1.0.0"))
      versionService.storeVersionOfComponent(valueOf("b", "1.0.0"))
      val questionAsked = new MutableBoolean(false)
      val upgrader = getIsolatedUpgrader(forceUpgrades = false, questionAsked, "yes")
      upgrader.autoUpgrade()
      questionAsked.booleanValue shouldBe true
      upgrader.getCount shouldBe 1
    }

    it("should not ask question when force upgrades is set to true") {
      versionService.storeVersionOfComponent(valueOf("a", "1.0.0"))
      versionService.storeVersionOfComponent(valueOf("b", "1.0.0"))
      val questionAsked = new MutableBoolean(false)
      val upgrader = getIsolatedUpgrader(forceUpgrades = true, questionAsked, "yes")
      upgrader.autoUpgrade()
      questionAsked.booleanValue shouldBe false
      upgrader.getCount shouldBe 0
    }

    it("should throw exception when rejecting upgrades") {
      versionService.storeVersionOfComponent(Version.valueOf("a", "1.0.0"))
      an[UpgradeRejectedException] shouldBe thrownBy {
        getIsolatedUpgrader(forceUpgrades = false, new MutableBoolean, "no").autoUpgrade()
      }
    }

    it("should upgrade to versions with data model") {
      val d_1 = new TestUpgrade(valueOf("d", "4.5.1"))
      val d_2 = new TestUpgrade(valueOf("d", "4.5.1#5"))
      val upgrader = new TestUpgrader(upgradeStrategy, true)
      versionService.storeVersionOfComponent(valueOf("d", "4.5.0"))
      upgrader.addComponent("d")
      upgrader.addUpgrade(d_2)
      upgrader.addUpgrade(d_1)
      upgrader.applyUpgrades()
      versionService.readVersionOfComponent("d") shouldBe d_2.upgradeVersion
    }

    it("should accept upgrades with different product versions and same data model version") {
      val upgrader = new TestUpgrader(upgradeStrategy, true)
      versionService.storeVersionOfComponent(valueOf("d", "4.5.0"))
      upgrader.addComponent("d")
      upgrader.addUpgrade(new TestUpgrade(valueOf("d", "4.5.1#5")))
      upgrader.addUpgrade(new TestUpgrade(valueOf("d", "4.5.2#5")))
      upgrader.applyUpgrades()
    }

    it("should fail if there are more than one upgrades with same data model version") {
      val upgrader = new TestUpgrader(upgradeStrategy, true)
      upgrader.setApplicationContext(new StaticApplicationContext)
      versionService.storeVersionOfComponent(valueOf("d", "4.5.0"))
      upgrader.addComponent("d")
      upgrader.addUpgrade(new TestUpgrade(valueOf("d", "4.5.1#5")))
      upgrader.addUpgrade(new TestUpgrade(valueOf("d", "4.5.1#5")))
      var caught = false
      try upgrader.applyUpgrades()
      catch {
        case e: UpgradeException =>
          caught = true
          e.getMessage shouldBe "Found 2 upgrades version d 4.5.1#5: TestUpgrade (d 4.5.1#5), TestUpgrade (d 4.5.1#5)"
      }
      assertTrue(caught)
      versionService.readVersionOfComponent("d") shouldBe valueOf("d", "4.5.0")
    }

    it("should not apply any upgrades with final version on installation") {
      val upgrader = new TestUpgrader(upgradeStrategy, true)
      versionService.storeVersionOfComponent(valueOf("d", "0.0.0"))
      upgrader.addComponent("d")
      upgrader.addUpgrade(new TestUpgrade(valueOf("d", "5.0.0")))
      upgrader.applyUpgrades()
      upgradesApplied.length shouldBe 0
      versionService.readVersionOfComponent("d") shouldBe valueOf("d", "5.0.0#0")
    }
  }

  private def getIsolatedUpgrader(forceUpgrades: Boolean, questionAsked: MutableBoolean, defaultAnswer: String) = {
    val upgrader = new TestUpgrader(upgradeStrategy, forceUpgrades) {
      override protected def read: String = {
        questionAsked.setValue(true)
        count += 1
        defaultAnswer
      }

      override

      def findUpgrades: util.Map[String, util.List[Upgrade]] = {
        val us = new util.HashMap[String, util.List[Upgrade]]
        us.put("a", util.Arrays.asList(a_2))
        us
      }
    }
    addDefaultUpgrades(upgrader)
    upgrader
  }

  private def addDefaultUpgrades(upgrader: Upgrader): Unit = {
    upgrader.addUpgrade(a_1)
    upgrader.addUpgrade(b_2)
    upgrader.addUpgrade(a_2)
    upgrader.addUpgrade(a_2_second_upgrade)
    upgrader.addUpgrade(a_3)
    upgrader.addUpgrade(c_1)
  }

  class TestUpgrade(var version: Version) extends Upgrade {
    @throws[UpgradeException]
    override def doUpgrade(): Boolean = {
      upgradesApplied += this
      true
    }

    override def upgradeVersion: Version = version

    override def toString: String = s"test upgrade: $version"
  }

  class TestUpgrader(val upgradeStrategy: UpgradeStrategy, val forceUpgrades: Boolean)
    extends Upgrader(upgradeStrategy, forceUpgrades, versionService) {
    addComponent("deployit")
    protected var count = 0

    def getCount: Int = count
  }

}
