package com.xebialabs.deployit.upgrade

import com.xebialabs.deployit.server.api.upgrade.RepositoryInitialization
import com.xebialabs.deployit.server.api.upgrade.Version.valueOf
import com.xebialabs.xlplatform.upgrade.RepositoryVersionService
import org.scalatest.funspec.AnyFunSpecLike
import org.scalatest.matchers.should.Matchers

import java._
import scala.collection.mutable
import scala.jdk.CollectionConverters._

trait RepositoryInitializerTest extends AnyFunSpecLike with Matchers {

  implicit val versionService: RepositoryVersionService
  implicit val initializationStrategy: InitializationStrategy

  private var a1: RepositoryInitialization = _
  private var a2: RepositoryInitialization = _
  private var a3: RepositoryInitialization = _
  private var b1: RepositoryInitialization = _
  private var c1: RepositoryInitialization = _

  private var initializationsApplied: mutable.ArrayBuffer[RepositoryInitialization] = _
  private var testRepositoryInitializer: RepositoryInitializer = _

  protected def clearVersionOfComponent(component: String): Unit

  def setup(): Unit = {
    initializationsApplied = mutable.ArrayBuffer.empty

    a1 = new TestInitialization("a", "a1")
    a2 = new TestInitialization("a", "a2")
    a3 = new XTestInitialization("a", "a3")
    b1 = new TestInitialization("b", "b1")
    c1 = new TestInitialization("c", "c1")

    testRepositoryInitializer = new TestRepositoryInitializer(
      initializationStrategy,
      Seq(a2, b1, a3, a1, c1),
      Map(
        "a" -> "1.0",
        "b" -> "2.0",
        "d" -> "0.1"
      ))
    clearVersionOfComponent("a")
    clearVersionOfComponent("b")
  }

  describe("Repository initializer") {
    it("should run component initializers") {
      versionService.readVersionOfComponent("a") shouldBe null
      versionService.readVersionOfComponent("b") shouldBe null

      testRepositoryInitializer.initializeComponents()

      initializationsApplied should have size 4
      initializationsApplied should contain.allOf(a1, a2, a3, b1)
    }

    it("should sort initializations alphabetically") {
      versionService.readVersionOfComponent("a") shouldBe null

      testRepositoryInitializer.initializeComponents()

      initializationsApplied.filter(_.getComponent == "a") should contain.allOf(a2, a1, a3)
    }

    it("should skip initialization if component version is not null") {
      versionService.storeVersionOfComponent(valueOf("a", "0.0.0"))

      versionService.readVersionOfComponent("a") should not be null
      versionService.readVersionOfComponent("b") shouldBe null

      testRepositoryInitializer.initializeComponents()

      initializationsApplied should have size 1
      initializationsApplied should contain(b1)
    }

    it("should skip initialization of uknown components") {
      testRepositoryInitializer.initializeComponents()

      initializationsApplied should not contain c1
    }
  }

  class TestInitialization(component: String, name: String) extends RepositoryInitialization {
    override def doInitialize(): Unit = initializationsApplied += this

    override def getComponent: String = component

    override def toString: String = name
  }

  class XTestInitialization(component: String, name: String) extends TestInitialization(component, name) {
  }

  class TestRepositoryInitializer(val initializationStrategy: InitializationStrategy,
                                  var initializations: Seq[RepositoryInitialization],
                                  var plugins: Map[String, String])
    extends RepositoryInitializer(initializationStrategy, versionService) {

    override private[upgrade] def findAndInstantiateApplicableInitializers: util.Collection[RepositoryInitialization] = initializations.asJava

    override private[upgrade] def getRegisteredPlugins: util.Set[String] = plugins.keySet.asJava
  }

}
