package com.xebialabs.deployit.setup

import java.io.File
import java.util.Objects

import com.xebialabs.deployit.setup.SetupHelperMethods.getBooleanResponse
import com.xebialabs.deployit.setup.UpgradeHelper._
import com.xebialabs.xlplatform.io.FolderChecksum.listAllFilesSorted
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils
import org.apache.commons.lang3.StringUtils

import scala.collection.JavaConverters._
import scala.collection.mutable.ListBuffer
import scala.util.{Failure, Success, Try}

object UpgradeHelper {
  case class DiscoveredPlugin(fileName: String, path: String)
}

class UpgradeHelper(previousInstallation: String, nonInteractiveMode: Boolean = false) extends Logging {

  /**
    * Copy files, directories passed to a method.
    * It also copied plugins directory skipping plugins that have newer version.
    *
    * @param files          list of files to copy(replace if exist) them one by one
    * @param dirs           list of directories to copy(replace if exist) as is
    * @param filesToMention list of files to mention that user should adapt and copy them manually
    */
  def copyData(files: java.util.List[String], dirs: java.util.List[String], filesToMention: java.util.List[String]): Unit = {
    validatePreviousInstallationLocation()
    println(s"\nGoing to copy configuration from previous installation [$previousInstallation].")

    val actions: List[CopyAction] =
      copyFiles(files.asScala.toList) ::
      copyDirectories(dirs.asScala.toList) :::
      copyPlugins()

    logger.info(s"Going to copy configuration from previous installation [$previousInstallation].")
    actions.foreach(_.execute())
    logger.info("Copy from previous installation completed.")

    println("\nCopy from previous installation completed.")
    mentionFiles(filesToMention.asScala.toList)
  }

  def mentionFiles(files: List[String]): Unit = {
    println("Please note the following:")
    println("- You should copy the database driver to the new installation manually.")
    files.foreach { file =>
      println(s"- You should adapt and copy [$file] to the new installation manually.")
    }
  }

  /**
    * Copy files from existing to the new installation.
    * Ask user for a confirmation if setup is in interactive mode.
    *
    * @param files list of file names relative to the installation folder
    */
  def copyFiles(files: List[String]): CopyAction = {
    val fileNames = files.mkString(", ")
    printIfInteractive(s"Do you want to copy files to the new installation?")
    printIfInteractive(s"Files to be copied [$fileNames]")
    askAndCopy(fileNames, "files") {
      files.foreach(tryCopy)
    }
  }

  /**
    * Copy directories from existing to the new installation.
    * Ask user for a confirmation if setup is in interactive mode.
    *
    * @param dirs list of directory names relative to the installation folder
    */
  def copyDirectories(dirs: List[String]): List[CopyAction] = {
    dirs.map { directory =>
      val previousDirectory = new File(previousInstallation, directory)
      printIfInteractive(s"Do you want to copy directory [$previousDirectory] to the new installation?")
      askAndCopy(directory, "directory") {
        tryCopy(directory)
      }
    }
  }

  type UpdatedPlugin = (String, String, String)

  private def validatePreviousInstallationLocation(): Unit = {
    val previousInstallationDir = new File(previousInstallation)
    if (!previousInstallationDir.exists || !previousInstallationDir.isDirectory) {
      throw new RuntimeException(s"Provided previous installation location [$previousInstallation] does not exist or is not a directory.")
    }
  }

  private def printIfInteractive(message: String): Unit = if (!nonInteractiveMode) println(message)

  private val supportedPluginExtensions = List(".jar", ".xldp", ".zip")

  /* SemVer regexp taken from https://github.com/semver/semver/blob/master/semver.md  */
  private val semver = """^(.*)((0|[1-9]\d*)\.(0|[1-9]\d*)\.(0|[1-9]\d*)(?:-((?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*)(?:\.(?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*))*))?(?:\+([0-9a-zA-Z-]+(?:\.[0-9a-zA-Z-]+)*))?)$""".r

  private[setup] def findMissingPlugins(oldPluginNames: List[DiscoveredPlugin],
                                        newPluginNames: List[DiscoveredPlugin],
                                        updatedPlugins: List[UpdatedPlugin]
                                       ): List[DiscoveredPlugin] = {
    def findMissing(names: List[DiscoveredPlugin]): List[DiscoveredPlugin] =
      names.filterNot(p =>
        updatedPlugins.map(_._1).exists(s => p.fileName.startsWith(s))
      )

    findMissing(oldPluginNames) diff findMissing(newPluginNames)
  }


  private[setup] def findUpdatedPlugins(oldPlugins: List[DiscoveredPlugin], currentPlugins: List[DiscoveredPlugin]): List[UpdatedPlugin] = {
    val versions = new ListBuffer[UpdatedPlugin]()

    def addIfUpdated(oldPlugin: Option[(String, String)], newPlugin: Option[(String, String)]): Unit = {
      (oldPlugin, newPlugin) match {
        case (Some((oldName, oldVersion)), Some((newName, newVersion))) =>
          if (Objects.equals(oldName, newName) && !Objects.equals(oldVersion, newVersion)) {
            versions += ((StringUtils.chop(oldName), oldVersion, newVersion))
          }
        case _ =>
      }
    }

    def extractVersion(pluginName: String, plugin: DiscoveredPlugin): Option[(String, String)] = pluginName match {
      case semver(commonName, version, _*) => Some((commonName, version))
      case _ => None
    }

    def comparePlugins(oldPlugin: DiscoveredPlugin, newPlugin: DiscoveredPlugin): Unit = {
      supportedPluginExtensions.foreach { extension =>
        if (oldPlugin.fileName.endsWith(extension) && newPlugin.fileName.endsWith(extension)) {
          val oldExtracted = extractVersion(oldPlugin.fileName.replace(extension, ""), oldPlugin)
          val newExtracted = extractVersion(newPlugin.fileName.replace(extension, ""), newPlugin)
          addIfUpdated(oldExtracted, newExtracted)
        }
      }
    }

    for (old <- oldPlugins) {
      for (current <- currentPlugins) {
        comparePlugins(old, current)
      }
    }
    versions.toList
  }

  class CopyAction(logMsg: String, action: => Unit) {
    def execute(): Unit = {
      logger.info(logMsg)
      action
    }
  }

  private def tryCopy(fileName: String): Unit = {
    val existingFile = new File(previousInstallation, fileName)
    val newFile = new File(fileName)

    def copyDirectory(): Unit = {
      Try(FileUtils.copyDirectory(existingFile, newFile)) match {
        case Success(_) => logger.info(s"Directory [$fileName] was copied to the new installation.")
        case Failure(exception) => logger.error(s"Error while copying directory [$fileName].", exception)
      }
    }

    def copyFile(): Unit = {
      Try(FileUtils.copyFile(existingFile, newFile)) match {
        case Success(_) => logger.info(s"File [$fileName] was copied to the new installation.")
        case Failure(exception) => logger.error(s"Error while copying file [$fileName].", exception)
      }
    }

    if (existingFile.isDirectory) copyDirectory() else copyFile()
  }

  private def copyPlugins(): List[CopyAction] = {
    val pluginsFolderName = "plugins"
    val previousPlugins = new File(previousInstallation, pluginsFolderName)
    logger.debug(s"Going to copy [$previousPlugins] from previous installation.")

    def listPlugins(file: File): List[DiscoveredPlugin] = listAllFilesSorted(file)
      .filter(p => supportedPluginExtensions.exists(e => p.getName.endsWith(e)))
      .map(plugin => DiscoveredPlugin(
        plugin.getName,
        file.toPath.relativize(plugin.toPath).toFile.getPath))

    val oldPlugins = listPlugins(previousPlugins)
    val newPlugins = listPlugins(new File(pluginsFolderName))
    val updatedPlugins = findUpdatedPlugins(oldPlugins, newPlugins)

    val missingPlugins: List[DiscoveredPlugin] = findMissingPlugins(oldPlugins, newPlugins, updatedPlugins)
    missingPlugins.map { plugin =>
      val pluginName = s"$pluginsFolderName/${plugin.path}"
      printIfInteractive(s"Plugin [$pluginName] is missing in the new installation. Do you want to copy it?")
      askAndCopy(pluginName, "plugin") {
        tryCopy(pluginName)
      }
    } :::
    updatedPlugins.map { case (name, oldVersion, newVersion) =>
      new CopyAction(s"Plugin [$name] has been updated [$oldVersion -> $newVersion]. Not copying it.", {})
    }
  }

  private def askAndCopy(name: String, fileType: String)(action: => Unit): CopyAction = {
    if (nonInteractiveMode) {
      new CopyAction(s"Copying $fileType [$name].", action)
    } else {
      if (getBooleanResponse(true)) {
        new CopyAction(s"User response was: [yes]. Copying $fileType [$name].", action)
      } else {
        new CopyAction(s"User response was: [no]. Not copying $fileType [$name].", {})
      }
    }
  }

}
