package com.xebialabs.deployit.setup;

import org.springframework.util.StringUtils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;

public class SetupHelperMethods {
    public static final String EXITSETUP_KEYWORD = "exitsetup";


    private SetupHelperMethods() {} // still no <code>static class</code> in Java


    public static void printEmptyLine() {
        System.out.println("");
    }

    public static String read() {
        BufferedReader stdin = new BufferedReader(new InputStreamReader(System.in));
        try {
            final String line = stdin.readLine();
            if (line != null) {
                String value = line.trim();
                if (EXITSETUP_KEYWORD.equals(value)) {
                    System.out.println("Exiting setup.");
                    System.exit(1);
                }
                return value;
            } else
                return line;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static boolean getBooleanResponse(boolean defaultValue) {
        String prompt = determinePrompt(defaultValue ? "yes" : "no");

        while (true) {
            System.out.print(prompt);
            String value = read();
            if ("".equals(value))
                return defaultValue;
            if ("yes".equalsIgnoreCase(value))
                return true;
            if ("no".equalsIgnoreCase(value))
                return false;
            System.out.println("Illegal value specified, type yes or no.");
        }
    }

    public static String getValidNotEmptyStringResponse(String defaultValue) {
        String prompt = determinePrompt(defaultValue);

        while (true) {
            System.out.print(prompt);
            String value = read();
            if ("".equals(value) && defaultValue != null)
                value = defaultValue;

            if (!"".equals(value) && value != null)
                return value;
            System.out.println("Value cannot be empty.");
        }
    }

    public static String getStringResponse(String defaultValue) {
        String prompt = determinePrompt(defaultValue);

        while (true) {
            System.out.print(prompt);
            String value = read();
            if ("".equals(value))
                value = defaultValue;
            if (value != null)
                return value;
        }
    }

    public static String getConfirmedPassword() {
        String password = null;
        while (true) {
            password = getPasswordResponse("New password: ");
            if (!password.isEmpty() && password.length() < 6) {
                System.err.println("Password is shorter than 6 characters, please try again.\n");
                continue;
            }
            String retypedPassword = getPasswordResponse("Re-type password: ");

            if (password.equals(retypedPassword)) {
                return password;
            }
            printEmptyLine();
            System.err.println("The passwords do not match. Please try again.");
        }

    }
    public static String getPasswordResponse(String prompt) {
        char[] password = System.console().readPassword(prompt);
        return new String(password);
    }

    public static int getValidIntegerResponse(int defaultValueAsAnInt) {
        String defaultValue = Integer.toString(defaultValueAsAnInt);
        String prompt = determinePrompt(defaultValue);

        while (true) {
            System.out.print(prompt);
            String value = read();
            if ("".equals(value))
                value = defaultValue;
            try {
                int returnedValue = Integer.parseInt(value);
                if (returnedValue > 0) {
                    return returnedValue;
                } else {
                    System.out.println("Illegal value specified. Number must be larger then 0.");
                }
            } catch (NumberFormatException nfe) {
                System.out.println("Illegal value specified. Must be a number.");
            }
        }
    }

    private static String retry(String prompt, String setUpMsg) {
        String value = null;
        System.out.println("User entered value is empty, setup might fail if you continue without entering values");
        System.out.println("Are you sure? Do you want to continue without entering the value? Options are yes or no.");
        boolean userSelection = getBooleanResponseForRetry();
        if (!userSelection) {
            printEmptyLine();
            System.out.println(setUpMsg);
            System.out.print(prompt);
            value = read();
            if (StringUtils.isEmpty(value)) {
                value = retry(prompt, setUpMsg);
            }
        }
        return value;
    }

    private static boolean getBooleanResponseForRetry() {
        String prompt = determinePrompt(null);

        while (true) {
            System.out.print(prompt);
            String value = read();
            if ("yes".equalsIgnoreCase(value))
                return true;
            if ("no".equalsIgnoreCase(value))
                return false;
            System.out.println("Illegal value specified, type yes or no.");
        }
    }

    public static String getStringResponseWithRetry(String message) {
        String prompt = determinePrompt(null);

        while (true) {
            System.out.print(prompt);
            String value = read();
            if (StringUtils.isEmpty(value)) {
                value = retry(prompt, message);
            }
            return value;
        }
    }

    public static String determinePrompt(String defaultValue) {
        String prompt;
        if (defaultValue != null)
            prompt = "[" + defaultValue + "]: ";
        else
            prompt = ": ";
        return prompt;
    }

}
