package com.xebialabs.deployit.jetty;

import java.io.IOException;
import java.io.PrintWriter;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.lang.String.format;

public class FlexRestCompatibilityServletResponse extends HttpServletResponseWrapper {
    private int status = 200;
    private ServletOutputStream outputStream;

    public FlexRestCompatibilityServletResponse(HttpServletResponse response) throws IOException {
        super(response);
        outputStream = response.getOutputStream();
    }

    public void writePreamble() throws IOException {
        outputStream.println("<response>");
        outputStream.println("<entity>");
    }

    public void writePostamble() throws IOException {
        outputStream.println("</entity>");
        outputStream.println(format("<status>%d</status>", status));
        outputStream.println("</response>");
    }



    @Override
    public ServletOutputStream getOutputStream() throws IOException {
        return outputStream;
    }

    @Override
    public void reset() {
        super.reset();
        try {
            writePreamble();
        } catch (IOException e) {
            throw new IllegalStateException("Cannot write flex compatibility preamble after resetting response.",e);
        }
    }

    @Override
    public PrintWriter getWriter() throws IOException {
        return new PrintWriter(outputStream);
    }

    @Override
    public void setStatus(final int sc) {
        int code = _determineCorrectStatusCode(sc);
        super.setStatus(code);
    }

    @Override
    @SuppressWarnings("deprecation")
    public void setStatus(final int sc, final String sm) {
        int code = _determineCorrectStatusCode(sc);
        super.setStatus(code, sm);
    }

    @Override
    public void sendError(final int sc) throws IOException {
        int code = _determineCorrectStatusCode(sc);
        super.sendError(code);
    }

    @Override
    public void sendError(final int sc, final String msg) throws IOException {
        int code = _determineCorrectStatusCode(sc);
        super.sendError(code, msg);
    }

    private int _determineCorrectStatusCode(final int sc) {
        addHeader(FlexRestCompatibilityFilter.X_ORIGINAL_STATUS_HEADER, Integer.toString(sc));
        status = sc;
        if (sc >= 400 || sc == 204) {
            logger.info("Hiding status code {} with {}", sc, 200);
            return 200;
        }
        return sc;
    }

    private static final Logger logger = LoggerFactory.getLogger(FlexRestCompatibilityServletResponse.class);

}
