package com.xebialabs.xlplatform

import com.typesafe.config._
import com.xebialabs.deployit.security.SecretKeyHolder
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.deployit.util.PasswordUtil.KEY_REPOSITORY_KEYSTORE_PASSWORD
import scala.collection.JavaConverters._
import scala.util.Try

package object config {

  sealed trait PathPart

  case class ObjectPath(path: String) extends PathPart

  case class Index(index: Int) extends PathPart

  implicit class ConfigPasswordUtils(val config: Config) extends AnyVal {
    private type PasswordApply = String => String

    private def buildPasswordsFromObject(pathParts: List[PathPart], obj: ConfigObject): Map[List[PathPart], String] =
      obj
        .entrySet()
        .asScala
        .foldLeft(Map[List[PathPart], String]()) { case (acc, entry) =>
          val parts = pathParts.lastOption match {
            case Some(ObjectPath(path)) => pathParts.dropRight(1) :+ ObjectPath(s"$path.${entry.getKey}")
            case _ => pathParts :+ ObjectPath(entry.getKey)
          }
          acc ++ buildPasswords(parts, entry.getValue)
        }

    private def buildPasswordsFromList(pathParts: List[PathPart], list: ConfigList): Map[List[PathPart], String] =
      list
        .asScala
        .toList
        .zipWithIndex
        .foldLeft(Map[List[PathPart], String]()) { case (acc, (value, index)) =>
          acc ++ buildPasswords(pathParts :+ Index(index), value)
        }

    private def describePathParts(pathParts: List[PathPart]): String =
      pathParts.foldLeft("") {
        case (acc, ObjectPath(path)) if acc.isEmpty => path
        case (acc, ObjectPath(path)) => s"$acc.$path"
        case (acc, Index(index)) => s"$acc[$index]"
        case (acc, _) => acc
      }

    private def buildPasswords(pathParts: List[PathPart], configValue: ConfigValue): Map[List[PathPart], String] =
      Try(configValue.valueType()).toOption.map {
        case ConfigValueType.STRING if isKeyForPassword(pathParts) =>
          Map(pathParts -> configValue.unwrapped().asInstanceOf[String])
        case ConfigValueType.OBJECT =>
          buildPasswordsFromObject(pathParts, configValue.asInstanceOf[ConfigObject])
        case ConfigValueType.LIST =>
          buildPasswordsFromList(pathParts, configValue.asInstanceOf[ConfigList])
        case _ if isKeyForPassword(pathParts) =>
          sys.error(s"""Property '${describePathParts(pathParts)}' must be a string value. Please enclose the value in double quotes (")""")
        case _ => Map[List[PathPart], String]()
      }.getOrElse(Map())

    private def passwords: Map[List[PathPart], String] =
      config
        .entrySet()
        .asScala
        .foldLeft(Map[List[PathPart], String]()) { case (acc, entry) =>
          acc ++ buildPasswords(List(ObjectPath(entry.getKey)), entry.getValue)
        }

    private def isKeyForPassword(pathParts: List[PathPart]) = pathParts.exists {
      case ObjectPath(k) =>
        k.contains("password") &&
          // ignore the repository keystore password, since it is always encrypted with the default key
          // furthermore, it should only be used to load the repository keystore at startup and never after that
          !KEY_REPOSITORY_KEYSTORE_PASSWORD.equals(k)
      case _ => false
    }

    private def applyForPath(remainingParts: List[PathPart], currentConfig: ConfigValue, value: AnyRef): ConfigValue =
      (currentConfig, remainingParts) match {
        case (configObject: ConfigObject, ObjectPath(path) :: Nil) =>
          val overrides = ConfigFactory.empty().withValue(path, ConfigValueFactory.fromAnyRef(value)).root()
          overrides.withFallback(configObject)
        case (configObject: ConfigObject, ObjectPath(path) :: tail) =>
          val cursor = configObject.toConfig.getValue(path)
          val newValue = applyForPath(tail, cursor, value)
          val overrides = ConfigFactory.empty().withValue(path, newValue).root()
          overrides.withFallback(configObject)
        case (configList: ConfigList, Index(index) :: tail) =>
          val patchedConfig = applyForPath(tail, configList.get(index), value)
          val patchedList = configList.asScala.toList.patch(index, patchedConfig :: Nil, 1)
          ConfigValueFactory.fromIterable(patchedList.asJava).withFallback(configList)
        case _ =>
          currentConfig
      }

    private def applyToPasswords(apply: PasswordApply) = {
      val appliedPasswords = passwords.mapValues(apply)
      appliedPasswords.foldLeft(config) { case (cfg, (path, password)) =>
        applyForPath(path, cfg.root(), password).asInstanceOf[ConfigObject].toConfig
      }
    }

    def encrypted(keyHolder: SecretKeyHolder): Config = {
      lazy val encrypter = new PasswordEncrypter(keyHolder)
      applyToPasswords(encrypter.ensureEncrypted)
    }

    def decrypted(keyHolder: SecretKeyHolder): Config = {
      lazy val encrypter = new PasswordEncrypter(keyHolder)
      applyToPasswords(encrypter.ensureDecrypted)
    }

    def allPasswordsEncrypted(keyHolder: SecretKeyHolder): Boolean = {
      lazy val encrypter = new PasswordEncrypter(keyHolder)
      passwords.values.forall(encrypter.isEncodedAndDecryptable)
    }

    def hasPlainPasswords(keyHolder: SecretKeyHolder): Boolean = !allPasswordsEncrypted(keyHolder)
  }

}