package com.xebialabs.xlplatform

import com.typesafe.config._
import com.xebialabs.deployit.security.SecretKeyHolder
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.deployit.util.PasswordUtil.KEY_REPOSITORY_KEYSTORE_PASSWORD

import scala.collection.JavaConverters._
import scala.util.Try

package object config {

  implicit class ConfigPasswordUtils(val config: Config) extends AnyVal {

    private type PasswordApply = String => String

    private def passwords = config.entrySet().asScala.map(e => e.getKey -> e.getValue).filter {
        case (k,v) if isKeyForPassword(k) && Try(v.valueType() == ConfigValueType.STRING).getOrElse(false) => true
        case (k,v) if isKeyForPassword(k) && Try(v.valueType() == ConfigValueType.STRING).isSuccess => sys.error( s"""Property '$k' must be a string value. Please enclose the value in double quotes (")""")
        case _ => false
      }.toMap.mapValues(_.unwrapped).asInstanceOf[Map[String, String]]

    private def isKeyForPassword(k: String) =
      k.contains("password") &&
        // ignore the repository keystore password, since it is always encrypted with the default key
        // furthermore, it should only be used to load the repository keystore at startup and never after that
        !KEY_REPOSITORY_KEYSTORE_PASSWORD.equals(k)

    private def applyToPasswords(apply: PasswordApply) = {
      val appliedPasswords = passwords.mapValues(apply)
      appliedPasswords.foldLeft(config) {
        case (cfg, (key, password)) =>
          cfg.withValue(key, ConfigValueFactory.fromAnyRef(password))
      }
    }

    def encrypted(keyHolder: SecretKeyHolder) = {
      lazy val encrypter = new PasswordEncrypter(keyHolder)
      applyToPasswords(encrypter.ensureEncrypted)
    }

    def decrypted(keyHolder: SecretKeyHolder) = {
      lazy val encrypter = new PasswordEncrypter(keyHolder)
      applyToPasswords(encrypter.ensureDecrypted)
    }

    def allPasswordsEncrypted(keyHolder: SecretKeyHolder) = {
      lazy val encrypter = new PasswordEncrypter(keyHolder)
      passwords.forall {
        case (k, v) => encrypter.isEncodedAndDecryptable(v)
      }
    }

    def hasPlainPasswords(keyHolder: SecretKeyHolder) = !allPasswordsEncrypted(keyHolder)
  }

}
