package com.xebialabs.deployit.spring;

import java.io.IOException;
import org.springframework.core.env.PropertySource;
import org.springframework.core.io.support.EncodedResource;
import org.springframework.core.io.support.PropertySourceFactory;
import org.springframework.util.StringUtils;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;

/**
 * Make Hocon configuration files usable from spring context/wiring/autowiring.
 *
 * Usage:
 *
 * <pre>
 * {@code
 *
 * @literal @Configuration
 * @literal @EnableOAuth2Client
 * @literal @PropertySource(factory = HoconPropertySourceFactory.class, value = "auth-oidc.conf")
 *  public class OpenIdConnectConfig {
 *    @literal @Value("${xl.security.auth.providers.oidc.clientId}")
 *     private String clientId;
 *     ...
 *  }
 * }
 * </pre>
 */
public class HoconPropertySourceFactory implements PropertySourceFactory {

    protected String getName(Config config) {
        if (config.origin().filename() != null) {
            return config.origin().filename();
        }
        if (config.origin().resource() != null) {
            return config.origin().resource();
        }
        if (config.origin().url() != null) {
            return config.origin().url().toString();
        }
        try {
            return config.origin().description();
        } catch (Exception e) {
            // seems like pretty much a can't happen
            return "typeSafe";
        }
    }

    @Override
    public PropertySource<?> createPropertySource(String name, EncodedResource resource) throws IOException {
        Config config = ConfigFactory.parseReader(resource.getReader()).withFallback(ConfigFactory.defaultReference()).resolve();
        String safeName = StringUtils.isEmpty(name) ? getName(config) : name;
        return new HoconConfigPropertySource(safeName, config);
    }
}
