package com.xebialabs.deployit.service.comparison;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import javax.xml.bind.DatatypeConverter;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.core.AbstractStringView;
import com.xebialabs.deployit.core.MapStringStringView;
import com.xebialabs.deployit.core.StringValue;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static java.util.stream.Collectors.toList;

@Component
public class Comparator {

    private static final String PASSWORD_OBFUSCATION = "********";

    public Map<String, List<String>> compare(ConfigurationItem reference, List<ConfigurationItem> entities) {
        ComputingLinkedMap comparison = new ComputingLinkedMap(entities.size() + 1);
        checkCiTypes(reference.getType(), entities);

        Descriptor descriptor = DescriptorRegistry.getDescriptor(reference.getType());
        int nrCi = 0;

        List<ConfigurationItem> entitiesToCompare = new ArrayList<>();
        entitiesToCompare.add(reference);
        entitiesToCompare.addAll(entities);
        for (ConfigurationItem compareEntity : entitiesToCompare) {
            writeFields(comparison, compareEntity, nrCi);
            writeProperties(comparison, compareEntity, descriptor, nrCi);
            nrCi++;
        }

        LinkedHashMap<String, List<String>> retval = new LinkedHashMap<>();
        for (String k : comparison.keySet()) {
            retval.computeIfAbsent(k, key -> new ArrayList<>()).addAll(comparison.get(k));
        }
        return retval;
    }

    @SuppressWarnings("unchecked")
    private void writeProperties(final ComputingLinkedMap comparison, final ConfigurationItem ci, final Descriptor descriptor, final int nrCi) {
        final Collection<PropertyDescriptor> nonHiddenDescriptors = descriptor.getPropertyDescriptors().stream().filter(pd -> !pd.isHidden()).collect(toList());
        for (PropertyDescriptor propertyDescriptor : nonHiddenDescriptors) {
            final String key = propertyDescriptor.getName();
            Object value = propertyDescriptor.get(ci);
            switch (propertyDescriptor.getKind()) {
                case BOOLEAN:
                case INTEGER:
                case STRING:
                    if (propertyDescriptor.isPassword()) {
                        value = PASSWORD_OBFUSCATION;
                    }
                case ENUM:
                case CI:
                    comparison.put(key, value != null ? value.toString() : null, nrCi);
                    break;
                case DATE:
                    if (value != null) {
                        Calendar cal = new GregorianCalendar();
                        cal.setTime((Date) propertyDescriptor.get(ci));
                        comparison.put(key, DatatypeConverter.printDateTime(cal), nrCi);
                    } else {
                        comparison.put(key, null, nrCi);
                    }
                case LIST_OF_STRING:
                    if (propertyDescriptor.isPassword()) {
                        value = Collections.singletonList(PASSWORD_OBFUSCATION);
                    }
                case LIST_OF_CI:
                case SET_OF_STRING:
                    if (propertyDescriptor.isPassword()) {
                        value = Collections.singleton(PASSWORD_OBFUSCATION);
                    }
                case SET_OF_CI:
                    if (value instanceof AbstractStringView) {
                        handleCollectionOfString(comparison, key, (Collection<String>) value, nrCi);
                    } else {
                        List<String> valueAsStrings = ((Collection<Object>) value).stream().map(Object::toString).collect(toList());
                        handleCollectionOfString(comparison, key, valueAsStrings, nrCi);
                    }
                    break;
                case MAP_STRING_STRING:
                    if (propertyDescriptor.isPassword()) {
                        Set<String> strings = ((Map<String, String>) value).keySet();
                        for (String string : strings) {
                            ((Map<String, String>) value).put(string, PASSWORD_OBFUSCATION);
                        }
                    }
                    handleMap(comparison, key, (Map<String, String>) value, nrCi);
                    break;
                default:
                    throw new IllegalStateException("Should not end up here!");
            }
        }
    }

    private void handleMap(ComputingLinkedMap comparison, String key, Map<String, String> value, int nrCi) {
        if (value != null) {
            // Order the keys in the Map
            SortedSet<String> set = new TreeSet<>(value.keySet());
            if (value instanceof MapStringStringView) {
                Map<String, StringValue> wrapped = ((MapStringStringView) value).getWrapped();
                for (String k : set) {
                    StringValue stringValue = wrapped.get(k);
                    comparison.put(key + ": " + k, stringValue.toPublicFacingValue(), nrCi);
                }
            } else {
                for (String k : set) {
                    comparison.put(key + ": " + k, value.get(k), nrCi);
                }
            }
        }
    }

    private void handleCollectionOfString(final ComputingLinkedMap comparison, final String key, final Collection<String> value, final int nrCi) {
        if (value != null) {
            List<String> list = new ArrayList<>(value);
            if (value instanceof AbstractStringView) {
                Collection<StringValue> wrapped = ((AbstractStringView<?>) value).getWrapped();
                list = wrapped.stream().map(StringValue::toPublicFacingValue).collect(toList());
            }
            Collections.sort(list);
            comparison.put(key, String.join(",", list), nrCi);
        }
    }

    private void writeFields(final ComputingLinkedMap comparison, final ConfigurationItem entity, int nrCi) {
        comparison.put("id", entity.getId(), nrCi);
        comparison.put("type", entity.getType().toString(), nrCi);
    }

    private void checkCiTypes(final Type referenceType, final Iterable<ConfigurationItem> entities) {
        for (ConfigurationItem entity : entities) {
            checkArgument(entity.getType().equals(referenceType), "Not all configuration items are of type %s", referenceType);
        }
    }

    @SuppressWarnings("serial")
    static class ComputingLinkedMap extends LinkedHashMap<String, List<String>> {
        private String[] strings;

        public ComputingLinkedMap(int listSize) {
            super();
            strings = new String[listSize];
            Arrays.fill(strings, "");
        }

        @Override
        public List<String> get(Object o) {
            List<String> value = super.get(o);
            if (value == null) {
                value = new ArrayList<>(Arrays.asList(strings));
                super.put((String) o, value);
            }

            return value;
        }

        public void put(String key, String value, int position) {
            List<String> values = get(key);
            values.remove(position);
            values.add(position, value);
        }
    }
}
