package com.xebialabs.xlplatform.distribution

import java.io.ByteArrayInputStream
import java.math.BigInteger
import java.nio.charset.StandardCharsets.UTF_8
import java.security._
import java.util.Base64

import org.bouncycastle.crypto.digests.SHA256Digest
import org.bouncycastle.crypto.generators.RSAKeyPairGenerator
import org.bouncycastle.crypto.params.{RSAKeyGenerationParameters, RSAKeyParameters}
import org.bouncycastle.crypto.signers.RSADigestSigner
import org.bouncycastle.crypto.util.{PublicKeyFactory, SubjectPublicKeyInfoFactory}

object RSAUtils {

  private final val random = new SecureRandom()

  // see org.bouncycastle.jce.provider.JDKKeyPairGenerator.RSA
  private final val defaultPublicExponent = BigInteger.valueOf(0x10001)
  private final val defaultTests = 12

  private def newKeyPairGenerator = new RSAKeyPairGenerator()

  private def newSigner = new RSADigestSigner(new SHA256Digest())

  def generateKeyPair(): RSAInfo = {
    val keyPairGenerator = newKeyPairGenerator
    keyPairGenerator.init(new RSAKeyGenerationParameters(defaultPublicExponent, random, 2048, defaultTests))
    val keyPair = keyPairGenerator.generateKeyPair
    RSAInfo(
      keyPair.getPrivate.asInstanceOf[RSAKeyParameters],
      SubjectPublicKeyInfoFactory.createSubjectPublicKeyInfo(keyPair.getPublic).getEncoded
    )
  }

  @throws[Exception]
  def sign(message: String, privateKey: RSAKeyParameters): String = {
    val signer = newSigner
    signer.init(true, privateKey)
    val bytes = message.getBytes(UTF_8)
    signer.update(bytes, 0, bytes.length)
    Base64.getEncoder.encodeToString(signer.generateSignature())
  }

  @throws[Exception]
  def verify(message: String, signature: String, publicKeyEncoded: Array[Byte]): Boolean = {
    val publicKeyParameter = PublicKeyFactory.createKey(new ByteArrayInputStream(publicKeyEncoded))
    val signer = newSigner
    signer.init(false, publicKeyParameter)
    val bytes = message.getBytes(UTF_8)
    signer.update(bytes, 0, bytes.length)
    val signatureBytes = Base64.getDecoder.decode(signature)
    signer.verifySignature(signatureBytes)
  }

}

case class RSAInfo(privateKey: RSAKeyParameters, publicKeyEncoded: Array[Byte])
