package com.xebialabs.deployit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;

import java.io.*;
import java.util.Properties;

public final class ServerConfigResource {

    private final Resource configResource;

    public ServerConfigResource(Resource resource) {
        configResource = resource;
    }

    public boolean exists() {
        return configResource.exists();
    }

    public String getName() {
        return configResource.getFilename();
    }

    public void readIntoConfiguration(ServerConfiguration serverConfiguration) {
        serverConfiguration.load(readConfigurationFile(configResource));
    }

    public ServerConfiguration loadConfig() {
        return loadConfig(true);
    }

    public ServerConfiguration loadConfig(boolean encryptedPasswords) {
        if (configResource.exists()) {
            ServerConfiguration serverConfiguration = readConfiguration(configResource, encryptedPasswords);
            if(!serverConfiguration.isValid()) {
                throw new Error("Configuration " + configResource.getFilename() + " is not valid please run with -setup option.");
            }

            return serverConfiguration;
        } else {
            throw new Error("Configuration " + configResource.getFilename()
                    + " does not exist, please run with -setup option before starting the server or command line interface.");
        }
    }

    static ServerConfiguration readConfiguration(Resource configResource, boolean encryptedPasswords) {
        ServerConfiguration serverConfiguration = new ServerConfiguration();
        serverConfiguration.setConfigResource(configResource);
        serverConfiguration.load(readConfigurationFile(configResource), encryptedPasswords);
        return serverConfiguration;
    }

    static Properties readConfigurationFile(Resource configResource) {
        Properties properties = new Properties();
        try (InputStream inputStream = configResource.getInputStream()) {
            properties.load(inputStream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return properties;
    }

    @Override
    public String toString() {
        return configResource.toString();
    }

    private static final Logger logger = LoggerFactory.getLogger(ServerConfigResource.class);
}
