/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit;

import com.xebialabs.deployit.conversion.StringCoder;
import com.xebialabs.deployit.security.PasswordGenerator;
import com.xebialabs.deployit.util.DeployitKeys;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.deployit.util.PasswordUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;

import java.io.File;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicReference;

import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;
import static com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty;
import static com.xebialabs.deployit.util.PasswordUtil.*;
import static com.xebialabs.deployit.util.PropertyUtil.*;
import static java.util.Map.Entry;

public class ServerConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(ServerConfiguration.class);

    // ******************
    // Jetty HTTP Server Defaults
    // ******************
    public static final String DEFAULT_WEBCONTENT_PACKAGE = "web";
    public static final String DEFAULT_WEB_CONTEXT_ROOT = "/";
    public static final String DEFAULT_ADMIN_PASSWORD = "admin";
    public static final String DEFAULT_DEPLOYIT_KEYS_KEYSTORE_PASSWORD = "deployit";
    public static final int DEFAULT_HTTP_PORT = 4516;
    public static final int DEFAULT_HTTPS_PORT = 4517;
    public static final int DEFAULT_MAX_THREADS = 150;
    public static final int DEFAULT_MIN_THREADS = 30;
    public static final int DEFAULT_IDLE_TIMEOUT = 30000;
    public static final int DEFAULT_STOP_TIMEOUT = 1000;
    public static final int DEFAULT_UPLOAD_FILE_BUFFER_SIZE = 256 * 1024;
    public static final String FILE_URL_PREFIX = "file://";
    public static final String DEFAULT_IMPORTABLE_PACKAGES_PATH = "importablePackages";
    public static final String DEFAULT_HTTP_BIND_ADDRESS = "0.0.0.0";
    public static final int DEFAULT_CLIENT_SESSION_TIMEOUT_MINUTES = 20;

    public static final int DEFAULT_CLIENT_SERVER_POLL_DELAY = 30000;
    public static final String DEFAULT_MAINTENANCE_FORBIDDEN_REQUESTS = "deployit/task:POST" +
            ",deployit/tasks/v2:POST" +
            ",deployit/control:POST" +
            ",xldeploy/task:POST" +
            ",xldeploy/tasks/v2:POST" +
            ",xldeploy/control:POST";

    public static final long DEFAULT_SPRING_CLOUD_RETRY_INITIAL_INTERVAL = 5000L;
    public static final int DEFAULT_SPRING_CLOUD_RETRY_MAX_ATTEMPTS = 30;
    public static final long DEFAULT_SPRING_CLOUD_RETRY_MAX_INTERVAL = 120000L;
    public static final double DEFAULT_SPRING_CLOUD_RETRY_MULTIPLIER = 1.1;

    public static final int DEFAULT_UPGRADE_BATCH_SIZE = 100;

    public static final String DEFAULT_SPRING_CLOUD_ENCRYPTION_KEY = "EncryptKey";

    // ******************
    // Property file keys
    // ******************
    public static final String KEY_USE_SSL = "ssl";
    public static final String KEY_USE_MUTUAL_SSL = "ssl.mutual";
    public static final String KEY_SSL_PROTOCOL = "ssl.protocol";
    public static final String KEY_KEYSTORE_PATH = "keystore.path";
    public static final String KEY_KEYSTORE_TYPE = "keystore.type";
    public static final String KEY_TRUSTSTORE_PATH = "truststore.path";
    public static final String KEY_SECURE_COOKIE_ENABLED = "secure.cookie.enabled";
    public static final String KEY_HTTP_BIND_ADDRESS = "http.bind.address";
    public static final String KEY_HTTP_PORT = "http.port";
    public static final String KEY_HTTP_CONTEXT_ROOT = "http.context.root";
    public static final String KEY_WELCOME_PAGE = "http.welcome.page";
    public static final String KEY_IMPORTABLE_PACKAGES_PATH = "importable.packages.path";
    public static final String KEY_MINIMUM_THREADS = "threads.min";
    public static final String KEY_MAXIMUM_THREADS = "threads.max";
    public static final String KEY_CLIENT_SESSION_TIMEOUT_MINUTES = "client.session.timeout.minutes";
    public static final String KEY_CLIENT_SESSION_REMEMBER_ENABLED = "client.session.remember.enabled";
    public static final String KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES = "client.automatically.map.all.deployables";
    public static final String KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_INITIAL = "client.automatically.map.all.deployables.on.initial";
    public static final String KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_UPDATE = "client.automatically.map.all.deployables.on.update";
    public static final String KEY_SERVER_RESOLVE_APPLICATION_DEPENDENCIES = "server.resolve.application.dependencies";
    public static final String KEY_CLIENT_SERVER_POLL_DELAY = "client.server_poll.delay";
    public static final String KEY_HIDE_INTERNALS = "hide.internals";
    public static final String KEY_SERVER_URL = "server.url";
    public static final String KEY_SERVER_MAPPING_OVERRIDE_DEPLOYED_FIELDS_ON_UPDATE = "server.mapping.override.deployed.fields.on.update";
    public static final String KEY_MAINTENANCE_FORBIDDEN_REQUESTS = "maintenance.forbidden_paths";
    public static final String KEY_UPLOAD_FILE_BUFFER_SIZE = "http.upload.buffer";
    public static final String KEY_HTTP_IDLE_SECONDS = "http.idle.seconds";
    public static final String KEY_JDBC_URL = "xl.repository.database.db-url";
    public static final String KEY_JDBC_DRIVER_CLASSNAME = "xl.repository.database.db-driver-classname";
    public static final String KEY_JDBC_USERNAME = "xl.repository.database.db-username";
    public static final String KEY_JDBC_PASSWORD = "xl.repository.database.db-password";

    public static final String KEY_TASK_IN_PROCESS_WORKER = "deploy.task.in-process-worker";
    public static final String KEY_UPGRADE_BATCH_SIZE = "upgrade.batch.size";
    public static final String KEY_SERVER_HOSTNAME = "server.hostname";
    public static final String KEY_SERVER_PORT = "server.port";

    // SPRING CLOUD
    public static final String KEY_SPRING_CLOUD_ENABLED = "xl.spring.cloud.enabled";
    public static final String KEY_SPRING_CLOUD_ENCRYPT_KEY = "xl.spring.cloud.encrypt.key";
    public static final String KEY_SPRING_CLOUD_URI = "xl.spring.cloud.uri";
    public static final String KEY_SPRING_CLOUD_EXTERNAL_CONFIG = "xl.spring.cloud.external-config";
    public static final String KEY_SPRING_CLOUD_RETRY_INITIAL_INTERVAL = "xl.spring.cloud.config.retry.initial.interval";
    public static final String KEY_SPRING_CLOUD_RETRY_MAX_ATTEMPTS = "xl.spring.cloud.config.retry.max.attempts";
    public static final String KEY_SPRING_CLOUD_RETRY_MAX_INTERVAL = "xl.spring.cloud.config.retry.max.interval";
    public static final String KEY_SPRING_CLOUD_RETRY_MULTIPLIER = "xl.spring.cloud.config.retry.multiplier";


    // PERMISSION SERVICE
    public static final String KEY_EXTERNAL_PERMISSION_SERVICE_SWITCH = "xl.permission.external-service";
    public static final String KEY_EXTERNAL_PERMISSION_SERVICE_URI = "xl.permission.external-service.uri";


    // ******************
    // Mandatory
    // ******************
    protected String keyStorePath;
    protected String keyStorePassword;
    protected String keyStoreKeyPassword;
    protected String keyStoreType;
    protected String trustStorePath;
    protected String trustStorePassword;
    private String httpBindAddress = DEFAULT_HTTP_BIND_ADDRESS;
    protected int httpPort;
    private String webContextRoot = DEFAULT_WEB_CONTEXT_ROOT;
    private String webWelcomePage;
    protected String importablePackagesPath;
    protected boolean ssl;
    protected boolean mutualSsl;
    protected String sslProtocol;
    protected int minThreads;
    protected String adminPassword;
    private boolean dirty;
    protected int maxThreads;
    private int clientSessionTimeoutMinutes;
    private boolean clientSessionRememberEnabled = true;
    private boolean hideInternals;
    private String serverUrl;
    private String repositoryKeyStorePassword;
    private Integer httpIdleSeconds = DEFAULT_IDLE_TIMEOUT / 1000;

    private String maintenanceForbiddenRequests = DEFAULT_MAINTENANCE_FORBIDDEN_REQUESTS;
    private int clientServerPollDelay = DEFAULT_CLIENT_SERVER_POLL_DELAY;

    private final Map<String, String> customPasswords = new HashMap<>();

    private boolean newConfiguration;

    private static final AtomicReference<ServerConfiguration> CONFIG = new AtomicReference<>();

    private File configFile;

    private Resource configResource;

    private final PasswordEncrypter defaultEncrypter = new PasswordEncrypter(DeployitKeys.DEFAULT_PASSWORD_ENCRYPTION_KEY);

    private boolean clientAutomaticallyMapAllDeployables = true;
    private boolean clientAutomaticallyMapAllDeployablesOnInitial = false;
    private boolean clientAutomaticallyMapAllDeployablesOnUpdate = false;

    private boolean serverResolveApplicationDependencies = true;
    private boolean serverMappingOverrideDeployedFieldsOnUpdate = true;

    private boolean secureCookieEnabled = false;

    private Integer uploadFileBufferSize;

    private String jdbcUrl;
    private String jdbcDriverClassName;
    private String jdbcUsername;
    private String jdbcPassword;

    private Boolean springCloudEnabled = false;
    private String springCloudEncryptKey;
    private Long springCloudRetryInitialInterval;
    private Integer springCloudRetryMaxAttempts;
    private Long springCloudRetryMaxInterval;
    private Double springCloudRetryMultiplier;
    private String springCloudUri;
    private boolean springCloudExternalConfig = false;

    private boolean externalPermissionService = false;
    private String externalPermissionServiceUri;
    private String serverHostname = "127.0.0.1";
    private int serverPort = 8180;

    private Integer upgradeBatchSize = DEFAULT_UPGRADE_BATCH_SIZE;

    public ServerConfiguration() {
        init();
    }

    public ServerConfiguration(boolean springCloudEnabled) {
        this.springCloudEnabled = springCloudEnabled;
        init();
    }

    public void init() {
        this.newConfiguration = true;
        setDefaults();
    }

    public static ServerConfiguration getInstance() {
        return CONFIG.get();
    }

    public static void setInstance(ServerConfiguration deployitConfig) {
        CONFIG.compareAndSet(null, deployitConfig);
    }

    public void setDefaults() {
        if (!springCloudEnabled) {
            setHttpBindAddress(DEFAULT_HTTP_BIND_ADDRESS);
            setImportablePackagesPath(DEFAULT_IMPORTABLE_PACKAGES_PATH);
            setMinThreads(DEFAULT_MIN_THREADS);
            setMaxThreads(DEFAULT_MAX_THREADS);
            setAdminPassword(DEFAULT_ADMIN_PASSWORD);
            setUploadFileBufferSize(DEFAULT_UPLOAD_FILE_BUFFER_SIZE);
            setClientSessionTimeoutMinutes(DEFAULT_CLIENT_SESSION_TIMEOUT_MINUTES);
        }
    }

    public void loadDefault() {
        load(new Properties(), false);
    }

    public void load(Properties properties) {
        load(properties, true);
    }

    public void load(Properties properties, boolean encryptedPasswords) {
        serverHostname = properties.getProperty(KEY_SERVER_HOSTNAME, serverHostname);
        serverPort = parseIntSilent(properties, KEY_SERVER_PORT, serverPort);
        ssl = "true".equalsIgnoreCase(properties.getProperty(KEY_USE_SSL, "false"));
        mutualSsl = "true".equalsIgnoreCase(properties.getProperty(KEY_USE_MUTUAL_SSL, "false"));
        sslProtocol = properties.getProperty(KEY_SSL_PROTOCOL);
        keyStorePath = properties.getProperty(KEY_KEYSTORE_PATH);
        keyStorePassword = properties.getProperty(KEY_KEYSTORE_PASSWORD);
        keyStoreKeyPassword = properties.getProperty(KEY_KEYSTORE_KEYPASSWORD);
        keyStoreType = properties.getProperty(KEY_KEYSTORE_TYPE);
        trustStorePath = properties.getProperty(KEY_TRUSTSTORE_PATH);
        trustStorePassword = properties.getProperty(KEY_TRUSTSTORE_PASSWORD);
        repositoryKeyStorePassword = properties.getProperty(KEY_REPOSITORY_KEYSTORE_PASSWORD);

        httpBindAddress = properties.getProperty(KEY_HTTP_BIND_ADDRESS, httpBindAddress);
        httpPort = parseIntSilent(properties, KEY_HTTP_PORT, httpPort);
        webContextRoot = properties.getProperty(KEY_HTTP_CONTEXT_ROOT, webContextRoot);
        webWelcomePage = properties.getProperty(KEY_WELCOME_PAGE);
        minThreads = parseIntSilent(properties, KEY_MINIMUM_THREADS, minThreads);
        maxThreads = parseIntSilent(properties, KEY_MAXIMUM_THREADS, maxThreads);

        adminPassword = properties.getProperty(KEY_ADMIN_PASSWORD, DEFAULT_ADMIN_PASSWORD);

        serverUrl = properties.getProperty(KEY_SERVER_URL, serverUrl);

        httpIdleSeconds = parseIntSilent(properties, KEY_HTTP_IDLE_SECONDS, DEFAULT_IDLE_TIMEOUT / 1000);

        jdbcUrl = properties.getProperty(KEY_JDBC_URL);
        jdbcDriverClassName = properties.getProperty(KEY_JDBC_DRIVER_CLASSNAME);
        jdbcUsername = properties.getProperty(KEY_JDBC_USERNAME);
        jdbcPassword = properties.getProperty(KEY_JDBC_PASSWORD);

        if (springCloudEnabled) {
            springCloudEncryptKey = properties.getProperty(KEY_SPRING_CLOUD_ENCRYPT_KEY, DEFAULT_SPRING_CLOUD_ENCRYPTION_KEY);
            springCloudEnabled = "true".equalsIgnoreCase(properties.getProperty(KEY_SPRING_CLOUD_ENABLED));
            springCloudRetryInitialInterval = parseLongSilent(properties, KEY_SPRING_CLOUD_RETRY_INITIAL_INTERVAL, DEFAULT_SPRING_CLOUD_RETRY_INITIAL_INTERVAL);
            springCloudRetryMaxInterval = parseLongSilent(properties, KEY_SPRING_CLOUD_RETRY_MAX_INTERVAL, DEFAULT_SPRING_CLOUD_RETRY_MAX_INTERVAL);
            springCloudRetryMaxAttempts = parseIntSilent(properties, KEY_SPRING_CLOUD_RETRY_MAX_ATTEMPTS, DEFAULT_SPRING_CLOUD_RETRY_MAX_ATTEMPTS);
            springCloudRetryMultiplier = parseDoubleSilent(properties, KEY_SPRING_CLOUD_RETRY_MULTIPLIER, DEFAULT_SPRING_CLOUD_RETRY_MULTIPLIER);
            springCloudUri = properties.getProperty(KEY_SPRING_CLOUD_URI);
            springCloudExternalConfig = "true".equalsIgnoreCase((properties.getProperty(KEY_SPRING_CLOUD_EXTERNAL_CONFIG, "false")));
        }

        if (!springCloudEnabled) {
            clientSessionTimeoutMinutes = parseIntSilent(properties, KEY_CLIENT_SESSION_TIMEOUT_MINUTES, DEFAULT_CLIENT_SESSION_TIMEOUT_MINUTES);
            clientAutomaticallyMapAllDeployables = "true".equalsIgnoreCase(properties.getProperty(KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES, "true"));
            clientAutomaticallyMapAllDeployablesOnInitial = "true".equalsIgnoreCase(properties.getProperty(KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_INITIAL, "false"));
            clientAutomaticallyMapAllDeployablesOnUpdate = "true".equalsIgnoreCase(properties.getProperty(KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_UPDATE, "false"));
            clientSessionRememberEnabled = "true".equalsIgnoreCase(properties.getProperty(KEY_CLIENT_SESSION_REMEMBER_ENABLED, "true"));
            clientServerPollDelay = parseIntSilent(properties, KEY_CLIENT_SERVER_POLL_DELAY, clientServerPollDelay);
            importablePackagesPath = properties.getProperty(KEY_IMPORTABLE_PACKAGES_PATH, importablePackagesPath);
            maintenanceForbiddenRequests = properties.getProperty(KEY_MAINTENANCE_FORBIDDEN_REQUESTS, maintenanceForbiddenRequests);
            serverResolveApplicationDependencies = "true".equalsIgnoreCase(properties.getProperty(KEY_SERVER_RESOLVE_APPLICATION_DEPENDENCIES, "true"));
            secureCookieEnabled = "true".equalsIgnoreCase(properties.getProperty(KEY_SECURE_COOKIE_ENABLED, "false"));
            serverMappingOverrideDeployedFieldsOnUpdate = "true".equalsIgnoreCase(properties.getProperty(KEY_SERVER_MAPPING_OVERRIDE_DEPLOYED_FIELDS_ON_UPDATE, "true"));
            upgradeBatchSize = parseIntSilent(properties, KEY_UPGRADE_BATCH_SIZE, DEFAULT_UPGRADE_BATCH_SIZE);
            uploadFileBufferSize = parseIntSilent(properties, KEY_UPLOAD_FILE_BUFFER_SIZE, DEFAULT_UPLOAD_FILE_BUFFER_SIZE);
            hideInternals = "true".equalsIgnoreCase(properties.getProperty(KEY_HIDE_INTERNALS, "false"));
        }

        externalPermissionService = "true".equalsIgnoreCase(properties.getProperty(KEY_EXTERNAL_PERMISSION_SERVICE_SWITCH, "false"));
        externalPermissionServiceUri = properties.getProperty(KEY_EXTERNAL_PERMISSION_SERVICE_URI);

        for (String key : properties.stringPropertyNames()) {
            if (PasswordUtil.isCustomPassword(key)) {
                String password = properties.getProperty(key);
                customPasswords.put(key, password);
            }
        }

        if (isNotEmpty(repositoryKeyStorePassword)) {
            if (defaultEncrypter.isEncodedAndDecryptable(repositoryKeyStorePassword)) {
                repositoryKeyStorePassword = defaultEncrypter.decrypt(repositoryKeyStorePassword);
            } else {
                logger.info("Read plaintext repository keystore password from the file, marking as dirty to save encrypted.");
                dirty = true;
            }
        }

        if (encryptedPasswords) {
            loadEncryptedPasswords();
        }

        this.newConfiguration = false;
    }

    public void loadEncryptedPasswords() {
        if (PasswordEncrypter.getInstance().isEncodedAndDecryptable(adminPassword)) {
            adminPassword = PasswordEncrypter.getInstance().decrypt(adminPassword);
        } else {
            logger.info("Read plaintext admin password from the file, marking as dirty to save encrypted.");
            dirty = true;
        }
        if (PasswordEncrypter.getInstance().isEncodedAndDecryptable(keyStoreKeyPassword)) {
            keyStoreKeyPassword = PasswordEncrypter.getInstance().decrypt(keyStoreKeyPassword);
        } else if (isNotEmpty(keyStoreKeyPassword)) {
            logger.info("Read plaintext keystore key password from the file, marking as dirty to save encrypted.");
            dirty = true;
        }
        if (PasswordEncrypter.getInstance().isEncodedAndDecryptable(keyStorePassword)) {
            keyStorePassword = PasswordEncrypter.getInstance().decrypt(keyStorePassword);
        } else if (isNotEmpty(keyStorePassword)) {
            logger.info("Read plaintext keystore password from the file, marking as dirty to save encrypted.");
            dirty = true;
        }
        if (PasswordEncrypter.getInstance().isEncodedAndDecryptable(trustStorePassword)) {
            trustStorePassword = PasswordEncrypter.getInstance().decrypt(trustStorePassword);
        } else if (isNotEmpty(trustStorePassword)) {
            logger.info("Read plaintext truststore password from the file, marking as dirty to save encrypted.");
            dirty = true;
        }
        if (PasswordEncrypter.getInstance().isEncodedAndDecryptable(jdbcPassword)) {
            jdbcPassword = PasswordEncrypter.getInstance().decrypt(jdbcPassword);
        } else if (isNotEmpty(jdbcPassword)) {
            logger.info("Read plaintext jdbc password from the file, marking as dirty to save encrypted.");
            dirty = true;
        }
        for (Entry<String, String> entry : customPasswords.entrySet()) {
            if (PasswordEncrypter.getInstance().isEncodedAndDecryptable(entry.getValue())) {
                customPasswords.put(entry.getKey(), PasswordEncrypter.getInstance().decrypt(entry.getValue()));
            } else if (isNotEmpty(entry.getValue())) {
                logger.info("Read plaintext '" + entry.getKey() + "' from the file, marking as dirty to save encrypted.");
                dirty = true;
            }
        }
    }

    public void save(Properties properties) {
        properties.setProperty(KEY_SERVER_HOSTNAME, serverHostname);
        properties.setProperty(KEY_SERVER_PORT, Integer.toString(serverPort));

        if (repositoryKeyStorePassword != null) {
            properties.setProperty(KEY_REPOSITORY_KEYSTORE_PASSWORD, defaultEncrypter.ensureEncrypted(getRepositoryKeyStorePassword()));
        }

        properties.setProperty(KEY_ADMIN_PASSWORD, PasswordEncrypter.getInstance().ensureEncrypted(adminPassword != null ? getAdminPassword() : DEFAULT_ADMIN_PASSWORD));

        if (serverUrl != null) {
            properties.setProperty(KEY_SERVER_URL, serverUrl);
        }

        for (Entry<String, String> entry : customPasswords.entrySet()) {
            properties.setProperty(entry.getKey(), PasswordEncrypter.getInstance().ensureEncrypted(entry.getValue()));
        }

        properties.setProperty(KEY_HTTP_BIND_ADDRESS, httpBindAddress);
        properties.setProperty(KEY_HTTP_CONTEXT_ROOT, webContextRoot);
        properties.setProperty(KEY_HTTP_PORT, Integer.toString(getHttpPort()));
        properties.setProperty(KEY_MAXIMUM_THREADS, Integer.toString(maxThreads));
        properties.setProperty(KEY_MINIMUM_THREADS, Integer.toString(minThreads));

        if (webWelcomePage != null) {
            properties.setProperty(KEY_WELCOME_PAGE, webWelcomePage);
        }

        saveSslProperties(properties);


        if (!springCloudEnabled) {
            properties.setProperty(KEY_CLIENT_SESSION_TIMEOUT_MINUTES, Integer.toString(clientSessionTimeoutMinutes));
            properties.setProperty(KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES, Boolean.toString(clientAutomaticallyMapAllDeployables));
            properties.setProperty(KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_INITIAL, Boolean.toString(clientAutomaticallyMapAllDeployablesOnInitial));
            properties.setProperty(KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_UPDATE, Boolean.toString(clientAutomaticallyMapAllDeployablesOnUpdate));
            properties.setProperty(KEY_CLIENT_SERVER_POLL_DELAY, Integer.toString(clientServerPollDelay));
            properties.setProperty(KEY_CLIENT_SESSION_REMEMBER_ENABLED, Boolean.toString(clientSessionRememberEnabled));

            properties.setProperty(KEY_HIDE_INTERNALS, Boolean.toString(hideInternals));
            properties.setProperty(KEY_IMPORTABLE_PACKAGES_PATH, getImportablePackagesPath());
            properties.setProperty(KEY_MAINTENANCE_FORBIDDEN_REQUESTS, maintenanceForbiddenRequests);
            properties.setProperty(KEY_SERVER_MAPPING_OVERRIDE_DEPLOYED_FIELDS_ON_UPDATE, Boolean.toString(serverMappingOverrideDeployedFieldsOnUpdate));
            properties.setProperty(KEY_SERVER_RESOLVE_APPLICATION_DEPENDENCIES, Boolean.toString(serverResolveApplicationDependencies));
            properties.setProperty(KEY_UPGRADE_BATCH_SIZE, getUpgradeBatchSize().toString());
            properties.setProperty(KEY_UPLOAD_FILE_BUFFER_SIZE, getUploadFileBufferSize().toString());

            saveDbProperties(properties);
        }

        saveCloudProperties(properties);
    }

    private void saveSslProperties(Properties properties) {
        if (keyStoreKeyPassword != null) {
            properties.setProperty(KEY_KEYSTORE_KEYPASSWORD, PasswordEncrypter.getInstance().ensureEncrypted(getKeyStoreKeyPassword()));
        }
        if (keyStorePassword != null) {
            properties.setProperty(KEY_KEYSTORE_PASSWORD, PasswordEncrypter.getInstance().ensureEncrypted(getKeyStorePassword()));
        }
        if (getKeyStorePath() != null) {
            properties.setProperty(KEY_KEYSTORE_PATH, getKeyStorePath());
        }
        if (keyStoreType != null) {
            properties.setProperty(KEY_KEYSTORE_TYPE, getKeyStoreType());
        }
        if (getSslProtocol() != null) {
            properties.setProperty(KEY_SSL_PROTOCOL, getSslProtocol());
        }
        if (getTrustStorePath() != null) {
            properties.setProperty(KEY_TRUSTSTORE_PATH, getTrustStorePath());
        }
        if (trustStorePassword != null) {
            properties.setProperty(KEY_TRUSTSTORE_PASSWORD, PasswordEncrypter.getInstance().ensureEncrypted(getTrustStorePassword()));
        }
        properties.setProperty(KEY_USE_SSL, Boolean.toString(isSsl()));
        properties.setProperty(KEY_USE_MUTUAL_SSL, Boolean.toString(isMutualSsl()));
    }

    private void saveDbProperties(Properties properties) {
        if (jdbcUrl != null) {
            properties.setProperty(KEY_JDBC_URL, jdbcUrl);
        }

        if (jdbcDriverClassName != null) {
            properties.setProperty(KEY_JDBC_DRIVER_CLASSNAME, jdbcDriverClassName);
        }

        if (jdbcUsername != null) {
            properties.setProperty(KEY_JDBC_USERNAME, jdbcUsername);
        }

        if (jdbcPassword != null) {
            properties.setProperty(KEY_JDBC_PASSWORD, PasswordEncrypter.getInstance().ensureEncrypted(jdbcPassword));
        }
    }

    private void saveCloudProperties(Properties properties) {
        if (springCloudEnabled) {
            updateProperty(properties, KEY_SPRING_CLOUD_URI, springCloudUri, this::setSpringCloudUri);
            updateBooleanProperty(properties, KEY_SPRING_CLOUD_EXTERNAL_CONFIG, springCloudExternalConfig, false, this::setSpringCloudExternalConfig);
            updateBooleanProperty(properties, KEY_SPRING_CLOUD_ENABLED, springCloudEnabled, true, this::setSpringCloudEnabled);

            updateLongProperty(properties, KEY_SPRING_CLOUD_RETRY_INITIAL_INTERVAL, springCloudRetryInitialInterval, DEFAULT_SPRING_CLOUD_RETRY_INITIAL_INTERVAL, this::setSpringCloudRetryInitialInterval);
            updateLongProperty(properties, KEY_SPRING_CLOUD_RETRY_MAX_INTERVAL, springCloudRetryMaxInterval, DEFAULT_SPRING_CLOUD_RETRY_MAX_INTERVAL, this::setSpringCloudRetryMaxInterval);
            updateIntProperty(properties, KEY_SPRING_CLOUD_RETRY_MAX_ATTEMPTS, springCloudRetryMaxAttempts, DEFAULT_SPRING_CLOUD_RETRY_MAX_ATTEMPTS, this::setSpringCloudRetryMaxAttempts);
            updateDoubleProperty(properties, KEY_SPRING_CLOUD_RETRY_MULTIPLIER, springCloudRetryMultiplier, DEFAULT_SPRING_CLOUD_RETRY_MULTIPLIER, this::setSpringCloudRetryMultiplier);
        }

        if (System.getenv("ENCRYPT_KEY") == null) {
            if (springCloudEncryptKey != null) {
                setSpringCloudEncryptKey(springCloudEncryptKey);
            }
            updateProperty(properties, KEY_SPRING_CLOUD_ENCRYPT_KEY, () -> springCloudEncryptKey,
                    this::generateCloudKey, this::setSpringCloudEncryptKey);
        }
    }

    public String generateCloudKey() {
        return PasswordGenerator.generateStrongPassword(15);
    }

    public boolean isValid() {
        boolean valid = true;
        if (isSsl()) {
            if (isBlank(getKeyStorePath())) {
                System.err.println(KEY_KEYSTORE_PATH + " must be set.");
                valid = false;
            } else if (!new File(getKeyStorePath()).exists()) {
                System.err.println(KEY_KEYSTORE_PATH + " must point to an existing keystore.");
                valid = false;
            }

            if (isMutualSsl()) {
                if (isBlank(getTrustStorePath())) {
                    System.err.println(KEY_TRUSTSTORE_PATH + " must be set.");
                    valid = false;
                } else if (!new File(getTrustStorePath()).exists()) {
                    System.err.println(KEY_TRUSTSTORE_PATH + " must point to an existing truststore.");
                    valid = false;
                }
            }
        }

        valid &= checkSet(getHttpBindAddress(), KEY_HTTP_BIND_ADDRESS);
        valid &= validateThatIntIsBiggerThen(KEY_HTTP_PORT, getHttpPort(), 0);
        valid &= checkSet(getWebContextRoot(), KEY_HTTP_CONTEXT_ROOT);
        if (!getWebContextRoot().startsWith("/")) {
            System.err.println(KEY_HTTP_CONTEXT_ROOT + " should start with a '/'");
            valid = false;
        }
        valid &= checkSet(getImportablePackagesPath(), KEY_IMPORTABLE_PACKAGES_PATH);
        valid &= validateThatIntIsBiggerThen(KEY_MINIMUM_THREADS, getMinThreads(), 0);
        valid &= validateThatIntIsBiggerThen(KEY_MAXIMUM_THREADS, getMaxThreads(), getMinThreads());
        valid &= validateThatIntIsBiggerThen(KEY_CLIENT_SESSION_TIMEOUT_MINUTES, getClientSessionTimeoutMinutes(), -1);

        return valid;
    }

    private static boolean checkSet(String value, String key) {
        if (isBlank(value)) {
            System.err.println(key + " must be set.");
            return false;
        }
        return true;
    }

    private static boolean validateThatIntIsBiggerThen(String key, int intToValidate, int shouldBeBiggerThen) {
        if (intToValidate <= shouldBeBiggerThen) {
            System.err.println(key + " is not valid, must be a number larger then " + shouldBeBiggerThen);
            return false;
        }
        return true;
    }

    public String getServerHostname() {
        return serverHostname;
    }

    public void setServerHostname(String serverHostname) {
        this.serverHostname = serverHostname;
    }

    public int getServerPort() {
        return serverPort;
    }

    public void setServerPort(int serverPort) {
        this.serverPort = serverPort;
    }

    public boolean isNewConfiguration() {
        return newConfiguration;
    }

    public void setSsl(boolean ssl) {
        this.ssl = ssl;
    }

    public boolean isSsl() {
        return ssl;
    }

    public void setMutualSsl(boolean mutualSsl) {
        this.mutualSsl = mutualSsl;
    }

    public boolean isMutualSsl() {
        return mutualSsl;
    }

    public String getKeyStorePath() {
        return keyStorePath;
    }

    public void setKeyStorePath(String keyStorePath) {
        this.keyStorePath = keyStorePath;
    }

    public String getKeyStorePassword() {
        return StringCoder.decode(keyStorePassword);
    }

    public void setKeyStorePassword(String keyStorePassword) {
        this.keyStorePassword = keyStorePassword;
    }

    public String getKeyStoreKeyPassword() {
        return StringCoder.decode(keyStoreKeyPassword);
    }

    public String getKeyStoreType() {
        return keyStoreType;
    }

    public void setKeyStoreType(String keyStoreType) {
        this.keyStoreType = keyStoreType;
    }

    public void setKeyStoreKeyPassword(String keyStoreKeyPassword) {
        this.keyStoreKeyPassword = keyStoreKeyPassword;
    }

    public String getTrustStorePath() {
        return trustStorePath;
    }

    public void setTrustStorePath(String trustStorePath) {
        this.trustStorePath = trustStorePath;
    }

    public String getTrustStorePassword() {
        return StringCoder.decode(trustStorePassword);
    }

    public void setTrustStorePassword(String trustStorePassword) {
        this.trustStorePassword = StringCoder.encode(trustStorePassword);
    }

    public String getRepositoryKeyStorePassword() {
        return StringCoder.decode(repositoryKeyStorePassword);
    }

    public void setRepositoryKeyStorePassword(String repositoryKeyStorePassword) {
        this.repositoryKeyStorePassword = StringCoder.encode(repositoryKeyStorePassword);
    }

    public String getHttpBindAddress() {
        return httpBindAddress;
    }

    public void setHttpBindAddress(String httpBindAddress) {
        this.httpBindAddress = httpBindAddress;
    }

    public int getHttpPort() {
        return httpPort;
    }

    public void setHttpPort(int httpPort) {
        this.httpPort = httpPort;
    }

    public String getImportablePackagesPath() {
        return importablePackagesPath;
    }

    public void setImportablePackagesPath(String setImportablePackagesPath) {
        this.importablePackagesPath = setImportablePackagesPath;
    }

    public String getSslProtocol() {
        return sslProtocol;
    }

    public void setSslProtocol(String sslProtocol) {
        this.sslProtocol = sslProtocol;
    }

    public int getMinThreads() {
        return minThreads;
    }

    public void setMinThreads(int minThreads) {
        this.minThreads = minThreads;
    }

    public int getMaxThreads() {
        return maxThreads;
    }

    public void setMaxThreads(int maxThreads) {
        this.maxThreads = maxThreads;
    }

    public void setClientSessionTimeoutMinutes(int clientSessionTimeoutMinutes) {
        this.clientSessionTimeoutMinutes = clientSessionTimeoutMinutes;
    }

    public int getClientSessionTimeoutMinutes() {
        return clientSessionTimeoutMinutes;
    }

    public boolean isClientSessionRememberEnabled() {
        return clientSessionRememberEnabled;
    }

    public boolean isDirty() {
        return dirty;
    }

    public String getAdminPassword() {
        return StringCoder.decode(adminPassword);
    }

    public void setAdminPassword(String adminPassword) {
        this.adminPassword = adminPassword;
    }

    public String getWebContextRoot() {
        return webContextRoot;
    }

    public void setWebContextRoot(String webContextRoot) {
        this.webContextRoot = webContextRoot;
    }

    public String getWebWelcomePage() {
        return webWelcomePage;
    }

    public void setWebWelcomePage(final String webWelcomePage) {
        this.webWelcomePage = webWelcomePage;
    }

    public boolean isHideInternals() {
        return hideInternals;
    }

    public void setHideInternals(boolean hideInternals) {
        this.hideInternals = hideInternals;
    }

    public File getFile() {
        return configFile;
    }

    public void setFile(File configFile) {
        this.configFile = configFile;
    }

    // Alternative to file. File has some limitations to work with.
    public Resource getConfigResource() {
        return configResource;
    }

    public void setConfigResource(Resource configResource) {
        this.configResource = configResource;
    }

    public String getServerUrl() {

        // Take the Server URL if defined in the properties file
        String url = serverUrl;

        // Derive it if not set
        if (url == null) {
            url = getDerivedServerUrl();
        }

        // Normalize so it ends with '/'
        if (url.charAt(url.length() - 1) != '/') {
            url += '/';
        }

        return url;
    }

    public void setServerUrl(String serverUrl) {
        this.serverUrl = serverUrl;
    }

    public String getDerivedServerUrl() {
        String hostname;
        try {
            hostname = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            logger.warn("Hostname could not be resolved from name service, server will use loopback address. Please make sure networking is properly configured (e.g. you can successfully ping your_hostname).", e);
            hostname = InetAddress.getLoopbackAddress().getHostAddress();
        }
        return (ssl ? "https" : "http") +
                "://" + hostname +
                ":" + httpPort +
                webContextRoot;
    }

    public String getCustomPassword(String propertyKey) {
        return customPasswords.get(propertyKey);
    }

    public boolean isClientAutomaticallyMapAllDeployables() {
        return clientAutomaticallyMapAllDeployables;
    }

    public void setClientAutomaticallyMapAllDeployables(boolean clientAutomaticallyMapAllDeployables) {
        this.clientAutomaticallyMapAllDeployables = clientAutomaticallyMapAllDeployables;
    }

    public boolean isClientAutomaticallyMapAllDeployablesOnInitial() {
        return clientAutomaticallyMapAllDeployablesOnInitial;
    }

    public void setClientAutomaticallyMapAllDeployablesOnInitial(boolean clientAutomaticallyMapAllDeployablesOnInitial) {
        this.clientAutomaticallyMapAllDeployablesOnInitial = clientAutomaticallyMapAllDeployablesOnInitial;
    }

    public boolean isClientAutomaticallyMapAllDeployablesOnUpdate() {
        return clientAutomaticallyMapAllDeployablesOnUpdate;
    }

    public void setClientAutomaticallyMapAllDeployablesOnUpdate(boolean clientAutomaticallyMapAllDeployablesOnUpdate) {
        this.clientAutomaticallyMapAllDeployablesOnUpdate = clientAutomaticallyMapAllDeployablesOnUpdate;
    }

    public boolean isServerMappingOverrideDeployedFieldsOnUpdate() {
        return serverMappingOverrideDeployedFieldsOnUpdate;
    }

    public void setServerMappingOverrideDeployedFieldsOnUpdate(boolean serverMappingOverrideDeployedFieldsOnUpdate) {
        this.serverMappingOverrideDeployedFieldsOnUpdate = serverMappingOverrideDeployedFieldsOnUpdate;
    }

    public boolean isServerResolveApplicationDependencies() {
        return serverResolveApplicationDependencies;
    }

    public void setServerResolveApplicationDependencies(boolean serverResolveApplicationDependencies) {
        this.serverResolveApplicationDependencies = serverResolveApplicationDependencies;
    }

    public String getMaintenanceForbiddenRequests() {
        return maintenanceForbiddenRequests;
    }

    public void setMaintenanceForbiddenRequests(final String maintenanceForbiddenRequests) {
        this.maintenanceForbiddenRequests = maintenanceForbiddenRequests;
    }

    public void setClientServerPollDelay(final int clientServerPollDelay) {
        this.clientServerPollDelay = clientServerPollDelay;
    }

    public int getClientServerPollDelay() {
        return clientServerPollDelay;
    }

    public Integer getUploadFileBufferSize() {
        return uploadFileBufferSize;
    }

    public void setUploadFileBufferSize(final Integer uploadFileBufferSize) {
        this.uploadFileBufferSize = uploadFileBufferSize;
    }

    public Integer getHttpIdleTimeoutMillis() {
        return httpIdleSeconds * 1000;
    }

    public boolean isSecureCookieEnabled() {
        return secureCookieEnabled;
    }

    public void setSecureCookieEnabled(boolean secureCookieEnabled) {
        this.secureCookieEnabled = secureCookieEnabled;
    }

    public void setJdbcUrl(String jdbcUrl) {
        this.jdbcUrl = jdbcUrl;
    }

    public String getJdbcUrl() {
        return jdbcUrl;
    }

    public void setJdbcDriverClassName(String jdbcDriverClassName) {
        this.jdbcDriverClassName = jdbcDriverClassName;
    }

    public String getJdbcDriverClassName() {
        return jdbcDriverClassName;
    }

    public void setJdbcUsername(String jdbcUsername) {
        this.jdbcUsername = jdbcUsername;
    }

    public String getJdbcUsername() {
        return jdbcUsername;
    }

    public void setJdbcPassword(String jdbcPassword) {
        this.jdbcPassword = jdbcPassword;
    }

    public String getJdbcPassword() {
        return jdbcPassword;
    }

    public String getSpringCloudUri() {
        return springCloudUri;
    }

    public void setSpringCloudUri(String springCloudUri) {
        this.springCloudUri = springCloudUri;
    }

    public boolean getSpringCloudExternalConfig() {
        return springCloudExternalConfig;
    }

    public void setSpringCloudExternalConfig(final boolean springCloudExternalConfig) {
        this.springCloudExternalConfig = springCloudExternalConfig;
    }

    public Long getSpringCloudRetryInitialInterval() {
        return springCloudRetryInitialInterval;
    }

    public void setSpringCloudRetryInitialInterval(Long springCloudRetryInitialInterval) {
        this.springCloudRetryInitialInterval = springCloudRetryInitialInterval;
    }

    public Integer getSpringCloudRetryMaxAttempts() {
        return springCloudRetryMaxAttempts;
    }

    public void setSpringCloudRetryMaxAttempts(Integer springCloudRetryMaxAttempts) {
        this.springCloudRetryMaxAttempts = springCloudRetryMaxAttempts;
    }

    public Long getSpringCloudRetryMaxInterval() {
        return springCloudRetryMaxInterval;
    }

    public void setSpringCloudRetryMaxInterval(Long springCloudRetryMaxInterval) {
        this.springCloudRetryMaxInterval = springCloudRetryMaxInterval;
    }

    public Double getSpringCloudRetryMultiplier() {
        return springCloudRetryMultiplier;
    }

    public void setSpringCloudRetryMultiplier(Double springCloudRetryMultiplier) {
        this.springCloudRetryMultiplier = springCloudRetryMultiplier;
    }

    public String getSpringCloudEncryptKey() {
        if (springCloudEncryptKey != null) {
            return PasswordEncrypter.getInstance().ensureDecrypted(springCloudEncryptKey);
        } else {
            return null;
        }
    }

    public void setSpringCloudEncryptKey(final String springCloudEncryptKeyUnencrypted) {
        this.springCloudEncryptKey = PasswordEncrypter.getInstance().ensureEncrypted(springCloudEncryptKeyUnencrypted);
    }

    public Boolean isSpringCloudEnabled() {
        return springCloudEnabled;
    }

    public void setSpringCloudEnabled(Boolean springCloudEnabled) {
        this.springCloudEnabled = springCloudEnabled;
    }

    public Integer getUpgradeBatchSize() {
        return upgradeBatchSize;
    }

    public void setUpgradeBatchSize(Integer upgradeBatchSize) {
        this.upgradeBatchSize = upgradeBatchSize;
    }

    public boolean isExternalPermissionService() {
        return externalPermissionService;
    }

    public void setExternalPermissionService(final boolean externalPermissionService) {
        this.externalPermissionService = externalPermissionService;
    }

    public String getExternalPermissionServiceUri() {
        return externalPermissionServiceUri;
    }

    public void setExternalPermissionServiceUri(final String externalPermissionServiceUri) {
        this.externalPermissionServiceUri = externalPermissionServiceUri;
    }
}

